import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";
import { nmapProfile } from '@/components/nmap/profiles/types';

export function useGetProfiles() {
  return useQuery<nmapProfile[], AxiosError>({
    queryKey: ['nmapProfiles'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/nmap/profile');
      return response.data;
    },
    staleTime: 5 * 60 * 1000, // 5 minutes
  });
}

export function useGetProfile(id: string) {
  return useQuery<nmapProfile, AxiosError>({
    queryKey: ['nmapProfile', id],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/nmap/profile/${id}`);
      return response.data;
    },
    enabled: !!id,
  });
}

export function useAddProfile() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Omit<nmapProfile, 'id' | 'user_id'> & { user_id?: number }) => {
      return axios.post(
        '/api/v1/nmap/profile',
        {
          user_id: data.user_id,
          name: data.name,
          parameters: data.parameters,
          description: data.description,
          tags: data.tags,
        },
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nmapProfiles'] });
      toast(t('Nmap Profile submitted.'));
    },
    onError: () => {
      toast(t('Failed to submit Nmap Profile'),
        { description: t('Please try again.')});
    }
  })
}

export function useDeleteProfiles() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (profileIds: number[]) => {
      return axios.delete(
        '/api/v1/nmap/profile',
        {
          data: { ids: profileIds },
        }
      );
    },
    onMutate: async (profileIds) => {
      const key = ['nmapProfiles'];
      await queryClient.cancelQueries({ queryKey: key });
      const previousprofiles = queryClient.getQueryData<nmapProfile[]>(key);
      queryClient.setQueryData(key, (old: nmapProfile[] = []) =>
        old.filter((profile) => !profileIds.includes(profile.id))
      );
      return { previousprofiles };
    },
    onSuccess: (_data, profileIds) => {
      const message = profileIds.length > 1
        ? t('Profiles deleted.')
        : t('Profile deleted.');
      toast(message);
    },
    onError: (_error, profileIds, context) => {
      queryClient.setQueryData(['nmapProfiles'], context?.previousprofiles)
      const errorMessage = profileIds.length > 1
        ? t('Failed to delete profiles.')
        : t('Failed to delete profile.');
      toast(errorMessage, {
        description: t('Please try again.'),
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['nmapProfiles'] });
    },
  });
}

export function useUpdateProfile() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Partial<nmapProfile> & { id: number }) => {
      return axios.put(
        `/api/v1/nmap/profile/${data.id}`,
        {
          name: data.name,
          parameters: data.parameters,
          description: data.description,
          tags: data.tags,
        },
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nmapProfiles'] });
      toast(t('Profile updated.'));
    },
    onError: () => {
      toast(t('Failed to update profile'),
        { description: t('Please try again.')});
    }
  });
}
