import * as React from "react";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { hostsColumns } from "@/components/nmap/scan/hostsColumns";
import { portsColumns } from "@/components/nmap/scan/portsColumns";
import { 
  Card, 
  CardContent, 
} from "@/components/ui/card";
import { ColumnDef } from "@tanstack/react-table";
import { useTranslation } from 'react-i18next';
import { Host } from "@/components/nmap/scan/types";
import InfoCard from "@/components/ui/InfoCard";
import {  
  Computer,
  Percent,
  Crosshair,
} from "lucide-react";
import { getScanHostsUpCount, getScanTargetCount } from "@/components/nmap/utils";
import { Skeleton } from "@/components/ui/skeleton"
  
type HostsProps = {
  data: any;
  isPending?: boolean;
  isError?: boolean;
};
  
export default function Hosts({ data, isPending, isError }: HostsProps) {
  const { t } = useTranslation();
  const [selectedHost, setSelectedHost] = React.useState<Host | null>(null);
  const handleRowClick = (host: Host) => setSelectedHost(host);

  if (isPending) {
    return (
      <Card>
        <CardContent>
          <div className="w-full flex gap-4 mb-4">
            <Skeleton className="h-45 w-1/3" />
            <Skeleton className="h-45 w-1/3" />
            <Skeleton className="h-45 w-1/3" />
          </div>
          <div className="flex">
            <Skeleton className="h-80 w-2/5 mr-4" />
            <Skeleton className="h-80 w-3/5 ml-4" />
          </div>
        </CardContent>
      </Card>
    );
  }

  if (isError || !data) {
    return <Card><CardContent>{t('Failed to load scan.')}</CardContent></Card>;
  }

  const hosts = data.json_results.host;
  const hostsArray = Array.isArray(hosts) ? hosts : hosts ? [hosts] : [];

  const totalTargets = getScanTargetCount(data);
  const hostsUp = getScanHostsUpCount(data);
  const percentUp = totalTargets > 0 ? ((hostsUp / totalTargets) * 100).toFixed(0) : '0';

  return (
    <Card>
      <CardContent className="flex flex-col gap-4">
        <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-4">
          <InfoCard 
            header={t("Targets scanned")}
            data={totalTargets}
            unit={totalTargets === 1 ? t("IP address") : t("IP addresses")}
            icon={<Crosshair />}
            tooltip={t("Number of targets scanned.")}
          />
          <InfoCard 
            header={t("Hosts found")}
            data={hostsUp}
            unit={hostsArray.length === 1 ? t("host") : t("hosts")}
            icon={<Computer />}
            tooltip={t("Number of hosts discovered in the scan.")}
          />
          <InfoCard
            header={t("Percentage up")}
            data={percentUp + "%"}
            icon={<Percent />}
            tooltip={t("Percentage of discovered hosts out of total scan targets.")}
          />
        </div>
        <div className="w-full flex">
          <div className="w-2/5 pr-4">
            <DataTable
              columns={hostsColumns as ColumnDef<Host, unknown>[]}
              data={hostsArray}
              onRowClick={handleRowClick}
              defaultColumnVisibility={{
                vendor: false,
                reason: false,
                status: false,
                hostname: false,
              }}
              checkActiveRow={(row) => {
                if (!selectedHost) return false;
              
                const getIPv4 = (host: Host | null | undefined): string | undefined =>
                  host
                    ? Array.isArray(host.address)
                      ? host.address.find(a => a.addrtype === "ipv4")?.addr
                      : host.address?.addrtype === "ipv4"
                      ? host.address.addr
                      : undefined
                    : undefined;
              
                const rowIP = getIPv4(row);
                const selectedIP = getIPv4(selectedHost);
              
                return !!(rowIP && selectedIP && rowIP === selectedIP);
              }}
            />
          </div>
          <div className="w-3/5 pl-4">
            {selectedHost ? (
              <DataTable
                columns={portsColumns(selectedHost)}
                data={
                  selectedHost && selectedHost.ports && selectedHost.ports.port
                    ? Array.isArray(selectedHost.ports.port)
                      ? selectedHost.ports.port
                      : [selectedHost.ports.port]
                    : []
                }
              />
            ) : (
              <div className="flex items-center justify-center h-full text-muted-foreground text-center">
                {t('Select a host to see ports and services.')}
              </div>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}