import * as React from "react";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { Card, CardContent } from "@/components/ui/card";
import { serviceColumns, ServiceEntry } from "./serviceColumns";
import { hostServiceColumns, HostServiceDetail } from "./hostServiceColumns";
import { useTranslation } from 'react-i18next';
import InfoCard from "@/components/ui/InfoCard";
import {  
  Activity,
  List,
} from "lucide-react"

type ServicesProps = {
  hosts: { host: any[] } | { host: any };
  isPending?: boolean;
  isError?: boolean;
}

export default function Services({ hosts, isPending, isError }: ServicesProps) {
  const { t } = useTranslation();

  function getHostIp(address: any): string {
    if (Array.isArray(address)) {
      const ipv4 = address.find((a) => a.addrtype === "ipv4");
      return ipv4 ? ipv4.addr : t("Unknown");
    } else if (address && address.addrtype === "ipv4") {
      return address.addr;
    }
    return t("Unknown");
  }

  if (isPending) {
    return <Card><CardContent>{t('Loading...')}</CardContent></Card>;
  }

  if (isError || !hosts) {
    return <Card><CardContent>{t('Failed to load scan.')}</CardContent></Card>;
  }

  const hostsArray = Array.isArray(hosts.host) ? hosts.host : hosts.host ? [hosts.host] : [];

  const serviceMap: Record<string, number> = {};
  const serviceToHostDetails: Record<string, HostServiceDetail[]> = {};
  let totalPorts = 0;

  (hostsArray).forEach((host: any) => {
    const hostIp = getHostIp(host.address);
    const hostStatus = host.status?.state || t("Unknown");
    const ports = host.ports?.port
      ? Array.isArray(host.ports.port)
      ? host.ports.port
      : [host.ports.port]
      : [];
    totalPorts += ports.length;
    ports.forEach((port: any) => {
      const serviceName = port.service?.name || t("Unknown");
      // Count for left table
      serviceMap[serviceName] = (serviceMap[serviceName] || 0) + 1;
      // Details for right table
      if (!serviceToHostDetails[serviceName]) serviceToHostDetails[serviceName] = [];
      serviceToHostDetails[serviceName].push({
        hostIp,
        port: port.portid,
        protocol: port.protocol,
        state: port.state?.state || t("Unknown"),
        status: hostStatus,
      });
    });
  });

  const services: ServiceEntry[] = Object.entries(serviceMap).map(([name, count]) => ({ name, count }));
  const [selectedService, setSelectedService] = React.useState<string | null>(null);

  return (
    <Card>
      <CardContent className="flex flex-col gap-4">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
          <InfoCard 
            header={t("Services found")}
            data={services.length}
            unit={services.length === 1 ? t("service") : t("services")}
            icon={<Activity />}
            tooltip={t("Number of unique services found.")}
          />
          <InfoCard
            header={t("Open ports")}
            data={totalPorts}
            unit={totalPorts === 1 ? t("port") : t("ports")}
            icon={<List />}
            tooltip={t("Total number of open ports found across all hosts.")}
          />
        </div>
        <div className="w-full flex">
          <div className="w-2/5 pr-4">
            <DataTable
                columns={serviceColumns}
                data={services}
                onRowClick={(row) => setSelectedService(row.name)}
                checkActiveRow={(row) => {
                  return selectedService === row.name;
                }}
              />
          </div>
          <div className="w-3/5 pl-4">
            {selectedService ? (
              <DataTable
                columns={hostServiceColumns}
                data={serviceToHostDetails[selectedService] || []}
              />
            ) : (
              <div className="flex items-center justify-center h-full text-muted-foreground text-center">
                {t('Select a service to see hosts.')}
              </div>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}