import { ColumnDef } from "@tanstack/react-table";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { Badge } from "@/components/ui/badge";
import { Host } from "@/components/nmap/scan/types";
import { useTranslation } from 'react-i18next';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Button } from "@/components/ui/button";
import { MoreHorizontal, GlobeLockIcon } from "lucide-react";
import SearchSuricata from "@/components/suricata/SearchSuricata";
import { useState } from "react";
import TextWithCopyButton from "@/components/ui/TextWithCopyButton";
import SearchRingBufferDialog from "@/components/wireshark/ringbuffer/SearchRingBufferDialog";
import SharkFin from "@/components/ui/icons/SharkFin";

export const hostsColumns: ColumnDef<Host, unknown>[] = [
  {
    id: "IPv4",
    accessorKey: "ipv4",
    header: ({ column }) => <DataTableColumnHeader column={column} title="IP Address" />,
    cell: ({ row }) => {
      const { t } = useTranslation();
      const address = Array.isArray(row.original.address) 
        ? row.original.address.find(a => a.addrtype === "ipv4")?.addr
        : row.original.address.addrtype === "ipv4" ? row.original.address.addr : undefined;
        return (
          address ? (
            <TextWithCopyButton text={address} />
          ) : t("Unknown")
        )
    },
  },
  {
    id: "MAC Address",
    accessorKey: "mac",
    header: ({ column }) => <DataTableColumnHeader column={column} title="MAC Address" />,
    cell: ({ row }) => {
      const { t } = useTranslation();
      const address = Array.isArray(row.original.address) 
        ? row.original.address.find(a => a.addrtype === "mac")?.addr
        : row.original.address.addrtype === "mac" ? row.original.address.addr : undefined;
      return (
        address ? (
          <TextWithCopyButton text={address} />
        ) : t("Unknown")
      )
    },
  },
  {
    id: "hostname",
    accessorKey: "hostname",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Hostname" />,
    cell: ({ row }) => {
      const { t } = useTranslation();
      if (Array.isArray(row.original.hostnames)) {
        return row.original.hostnames.map(h => h.hostname.name).join(", ") || t("Unknown");
      }
      if (row.original.hostnames?.hostname) {
        return row.original.hostnames.hostname.name || t("Unknown");
      }
      return t("Unknown");
    },
  },
  {
    id: "status",
    accessorKey: "status",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Status" />,
    cell: ({ row }) => (
      <Badge variant={row.original.status.state === "up" ? "default" : "destructive"}>
        {row.original.status.state.toUpperCase()}
      </Badge>
    ),
  },
  {
    id: "vendor",
    accessorKey: "vendor",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Vendor" />,
    cell: ({ row }) => {
      const { t } = useTranslation();
      const vendor = Array.isArray(row.original.address) 
        ? row.original.address.find(a => a.addrtype === "mac")?.vendor
        : row.original.address.addrtype === "mac" ? row.original.address.vendor : undefined;
      return vendor || t("Unknown");
    },
  },
  {
    id: "reason",
    accessorKey: "reason",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Reason" />,
    cell: ({ row }) => row.original.status.reason,
  },
  {
    id: "actions",
    cell: ({ row }) => {
      const { t } = useTranslation();
      const [searchSuricataDialogOpen, setSearchSuricataDialogOpen] = useState(false);
      const [searchWiresharkDialogOpen, setSearchWiresharkDialogOpen] = useState(false);

      const ipv4 = Array.isArray(row.original.address)
        ? row.original.address.find((a) => a.addrtype === "ipv4")?.addr
        : row.original.address?.addrtype === "ipv4"
        ? row.original.address.addr
        : undefined;

      return (
        <>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" className="h-8 w-8 p-0">
                <span className="sr-only">{t('Open menu')}</span>
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="center">
              <DropdownMenuItem onSelect={() => {setSearchSuricataDialogOpen(true)}}>
                <GlobeLockIcon />
                {t("Search Suricata")}
              </DropdownMenuItem>
              <DropdownMenuItem onSelect={() => {setSearchWiresharkDialogOpen(true)}}>
                <SharkFin />
                {t("Search")} Wireshark
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>

          <SearchSuricata
            open={searchSuricataDialogOpen}
            onOpenChange={setSearchSuricataDialogOpen}
            srcIp={ipv4}
            destIp={ipv4}
          />
          <SearchRingBufferDialog
            open={searchWiresharkDialogOpen}
            onOpenChange={setSearchWiresharkDialogOpen}
            src={ipv4}
            dst={ipv4}
          />

        </>
      )
    },
  },
]; 