import { ColumnDef } from "@tanstack/react-table";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem } from "@/components/ui/dropdown-menu";
import { MoreHorizontal, GlobeLockIcon } from "lucide-react";
import { useTranslation } from "react-i18next";
import { Host, Port } from "@/components/nmap/scan/types";
import { useState } from "react";
import SearchSuricata from "@/components/suricata/SearchSuricata";
import SearchRingBufferDialog from "@/components/wireshark/ringbuffer/SearchRingBufferDialog";
import SharkFin from "@/components/ui/icons/SharkFin";

export const portsColumns = (
  host: Host,
) : ColumnDef<Port>[] => [
  {
    id: "state",
    accessorFn: row => row.state?.state || "",
    header: ({ column }) => <DataTableColumnHeader column={column} title="State" />,
    cell: ({ row }) => (
      <Badge variant={row.original.state?.state === "open" ? "default" : "secondary"}>
        {row.original.state?.state.toUpperCase()}
      </Badge>
    ),
  },
  {
    accessorKey: "portid",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Port" />,
    cell: ({ row }) => row.original.portid,
  },
  {
    accessorKey: "protocol",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Protocol" />,
    cell: ({ row }) => row.original.protocol,
  },
  {
    accessorKey: "service",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Service" />,
    cell: ({ row }) => row.original.service?.name,
  },
  {
    accessorKey: "version",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Version" />,
    cell: ({ row }) => row.original.service?.version || "-",
  },
  {
    id: "actions",
    cell: ({ row }) => {
      const { t } = useTranslation();
      const [searchSuricataDialogOpen, setSearchSuricataDialogOpen] = useState(false);
      const [searchWiresharkDialogOpen, setSearchWiresharkDialogOpen] = useState(false);
  
      const ipv4 = Array.isArray(host.address)
        ? host.address.find((a) => a.addrtype === "ipv4")?.addr
        : host.address?.addrtype === "ipv4"
        ? host.address.addr
        : undefined;
  
      return (
        <>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" className="h-8 w-8 p-0">
                <span className="sr-only">{t("Open menu")}</span>
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="center">
              <DropdownMenuItem onSelect={() => {setSearchSuricataDialogOpen(true)}}>
                <GlobeLockIcon />
                {t("Search Suricata")}
              </DropdownMenuItem>
              <DropdownMenuItem onSelect={() => {setSearchWiresharkDialogOpen(true)}}>
                <SharkFin />
                {t("Search")} Wireshark
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
  
          <SearchSuricata
            open={searchSuricataDialogOpen}
            onOpenChange={setSearchSuricataDialogOpen}
            srcIp={ipv4}
            destIp={ipv4}
            srcPort={Number(row.original.portid) || undefined}
            destPort={Number(row.original.portid) || undefined}
            protocol={row.original.protocol}
          />
          <SearchRingBufferDialog
            open={searchWiresharkDialogOpen}
            onOpenChange={setSearchWiresharkDialogOpen}
            src={ipv4}
            dst={ipv4}
            srcPort={Number(row.original.portid) || undefined}
            dstPort={Number(row.original.portid) || undefined}
            protocol={row.original.protocol}
          />
        </>
      );
    },
  },  
]; 