import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import React from 'react';
import { columns } from "@/components/nmap/scans/columns";
import AddNmapScanDialog from "@/components/nmap/scans/AddNmapScanDialog";
import DeleteNmapScanDialog from "@/components/nmap/scans/DeleteNmapScanDialog";
import { useNmapIntegrationUpdates } from "@/components/nmap/NmapIntegrationUpdatesProvider";
import { useTranslation } from 'react-i18next';
import { nmapScan } from "@/components/nmap/scans/types";
import { Plus } from "lucide-react";
import { useAuth } from "@/contexts/AuthContext";

type ScansProps = {
  data: nmapScan[] | undefined;
  isLoading: boolean;
}

export default function Scans({ data, isLoading }: ScansProps) {
  const { t } = useTranslation();
  const { mergeScan } = useNmapIntegrationUpdates();
  const mergedData = (data || []).map(mergeScan);
  const [deleteDialogOpen, setDeleteDialogOpen] = React.useState(false);
  const [idsToDelete, setIdsToDelete] = React.useState<number[]>([]);
  const { user } = useAuth();

  const handleDelete = (id: number) => {
    setIdsToDelete([id]);
    setDeleteDialogOpen(true);
  }

  const actions = [
    {
      label: "Delete",
      value: "delete",
      onClick: async (selectedRows: any) => {
        setIdsToDelete(selectedRows.map((row: any) => row.id));
        setDeleteDialogOpen(true);
      }
    }
  ]

  return (
    <>
      <div className="pb-1 flex">
        {!!user?.is_admin &&
          <AddNmapScanDialog>
            <Button>
              <Plus />
              {t('Create Scan')}
            </Button>
          </AddNmapScanDialog>
        }
      </div>
      {!isLoading && 
        <DataTable 
          columns={columns(
            handleDelete,
          )} 
          data={mergedData} 
          sorting={[{ id: "Started At", desc: true }]}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultColumnVisibility={{
            'Finished At': false,
          }}
        />
      }
      {isLoading && <DataTableSkeleton />}
      <DeleteNmapScanDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      >
        <span />
      </DeleteNmapScanDialog>
    </>
  )
}