import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";
import { nmapScan } from '@/components/nmap/scans/types';

export function useDeleteScans() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (scanIds: number[]) => {
      return axios.delete(
        '/api/v1/nmap',
        {
          data: { ids: scanIds },
        }
      );
    },
    onMutate: async (scanIds) => {
      const key = ['nmapScans'];
      await queryClient.cancelQueries({ queryKey: key });
      const previousScans = queryClient.getQueryData<nmapScan[]>(key);
      queryClient.setQueryData(key, (old: nmapScan[] = []) =>
        old.filter((scan) => !scanIds.includes(scan.id))
      );
      return { previousScans };
    },
    onSuccess: (_data, scanIds) => {
      const message = scanIds.length > 1
        ? t('Nmap scans deleted.')
        : t('Nmap scan deleted.');
      toast(message);
    },
    onError: (_error, scanIds, context) => {
      queryClient.setQueryData(['nmapScans'], context?.previousScans)
      const errorMessage = scanIds.length > 1
        ? t('Failed to delete Nmap scans.')
        : t('Failed to delete Nmap scan.');
      toast(errorMessage, {
        description: t('Please try again.'),
      });
    },
    onSettled: () => {
      queryClient.refetchQueries({ queryKey: ['ndiffs'] });
      queryClient.invalidateQueries({ queryKey: ['nmapScans'] });
    },
  });
}

export function useDownloadScan() {
  return useMutation({
    mutationFn: async (filename: string) => {
      const response = await axios.get(
        `/api/v1/nmap/download/${filename}`,
        { responseType: 'blob' }
      );
      return response.data;
    },
  });
}

export function useGetScans() {
  return useQuery<any[], AxiosError>({
    queryKey: ['nmapScans'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/nmap');
      return response.data;
    },
    staleTime: 5 * 60 * 1000, // 5 minutes
  });
}

export function useAddScan() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Omit<nmapScan, 'id' | 'user_id' | 'created_at' | 'updated_at' | 'finished_at' | 'status' | 'output_filename' | 'scan_results'>) => {
      return axios.post(
          '/api/v1/nmap',
          {
            title: data.title,
            scan_parameters: data.scan_parameters,
            nmap_profile_id: data.nmap_profile_id ?? null,
          },
        );
      },
      onSuccess: () => {
        queryClient.invalidateQueries({ queryKey: ['nmapScans'] });
        toast(t('Nmap scan submitted.'),
          { description: t('Access your new scan via My Scans.') });
      },
      onError: () => {
        toast(t('Failed to submit Nmap scan.'),
          { description: t('Please try again.') });
      },
  });
}

export function useGetScan(id: string) {
  return useQuery({
    queryKey: ['nmapScan', id],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/nmap/${id}`);
      return response.data;
    },
    enabled: !!id,
  });
}

export function useStopScan() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.post(`/api/v1/nmap/${id}/stop`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nmapScans'] });
      toast.success(t('Scan stopped.'), {
        description: t('The scan has been terminated.'),
      });
    },
    onError: () => {
      toast.error(t('Failed to stop scan.'), {
        description: t('Please try again.'),
      });
    },
  });
}