import React, { useState } from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Checkbox } from "@/components/ui/checkbox";
import { useForm, SubmitHandler } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { useCreateSource, useEditSource } from "@/components/source/queries";
import type { Source } from "./types";
import { z } from "zod";
import { zodResolver } from '@hookform/resolvers/zod'


// Helper function to extract lifetime value and unit for form display
const parseLifetime = (lifetime?: string) => {
  if (!lifetime) return { value: "24", unit: "H" };
  const match = lifetime.match(/^(\d+)([A-Z])$/);
  if (match) {
    return { value: match[1], unit: match[2] };
  }
  return { value: "24", unit: "H" };
};

const sourceSchema = z.object({
    id: z.coerce.number().optional(),
    name: z.string().min(1, "Name is required"),
    addresses: z.string().optional(),
    port: z.coerce.number().min(1,"Port is required"),
    flowtype:z.string().optional(),
    abnormal_behavior: z.any().optional(),
    lifetimeValue:z.string().min(1,"Lifetime is required"),
    lifetimeUnit:z.string().min(1,"Choose a unit"),
});

type sourceValues= z.infer<typeof sourceSchema>;

export default function AddEditSourceDialog({ 
    source, 
    children,
    open,
    onOpenChange
}: { 
    source?: Source;
    children?: React.ReactNode;
    open?: boolean;
    onOpenChange?: (open: boolean) => void;
}) {
    const [internalOpen, setInternalOpen] = useState(false);
    const { t } = useTranslation();
    const createSource = useCreateSource();
    const updateSource = useEditSource();

    // Use external state if provided, otherwise use internal state
    const isOpen = open !== undefined ? open : internalOpen;
    const handleOpenChange = onOpenChange || setInternalOpen;

    const flowTypes = ["NetFlow", "sFlow", "jFlow", "IPFIX"];
    const lifetimeUnits = [
        { key: "H", label: "Hours" },
        { key: "D", label: "Days" },
        { key: "W", label: "Weeks" }
    ];

    // Parse lifetime for form display
    const { value: lifetimeValue, unit: lifetimeUnit } = parseLifetime(source?.lifetime);

    const form = useForm<sourceValues>({
        resolver: zodResolver(sourceSchema),
        defaultValues: {
            id: source?.id ?? undefined,
            name: source?.name ?? "",
            addresses: source?.addresses ?? "",
            port: source?.port ?? 0,
            flowtype: source?.flowtype ?? "netflow",
            lifetimeValue: lifetimeValue,
            lifetimeUnit: lifetimeUnit,
            abnormal_behavior: source?.abnormal_behavior ?? true,
        },
    });

    const onSubmit: SubmitHandler<sourceValues> = (data) => {
        // Combine lifetime value and unit back into single field
        const submitData: Source = {
            ...data,
            id: data.id ?? -1, 
            lifetime: `${data.lifetimeValue}${data.lifetimeUnit}`,
            port: Number(data.port),
        };

        if (source) {
            updateSource.mutate(submitData, {
                onSuccess: () => {
                    handleOpenChange(false);
                    form.reset();
                }
            });
        } else {
            createSource.mutate(submitData, {
                onSuccess: () => {
                    handleOpenChange(false);
                    form.reset();
                }
            });
        }
    };
    
    const handleDialogChange = (isOpen: boolean) => {
        handleOpenChange(isOpen);
        if (!isOpen) {
            form.reset();
        }
    };

    return (
        <Dialog open={isOpen} onOpenChange={handleDialogChange}>
            {children && (
                <DialogTrigger asChild>
                    {children}
                </DialogTrigger>
            )}
            <DialogContent className="sm:max-w-[800px] p-0">
                <DialogHeader className="bg-card p-6">
                    <DialogTitle>
                        {source ? t('Edit Flow Source') : t('Create Flow Source')}
                    </DialogTitle>
                    <DialogDescription>
                        {t('When adding a new source, make sure you set up the source to send flow data to your NNA installation IP address at the port you specify below to receive data.')}
                    </DialogDescription>
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)}>
                        <div className="flex flex-col gap-4 p-6">
                            <div className="mb-4">
                                <FormField 
                                    control={form.control}
                                    name="name"
                                    render={({ field }) => (
                                        <FormItem  className="flex items-center w-full flex-wrap gap-4">
                                            <div className="space-y-1 md:w-2/3">
                                                <FormLabel>{t('Flow Source Name')}</FormLabel>
                                                <FormDescription>{t('Name of the flow data collector, must be a unique name.')}</FormDescription>
                                            </div>
                                            <div className="relative flex-1 w-full">
                                                <FormControl className="flex-1">
                                                    <div className="col-span-2 w-full">
                                                        <Input {...field} />
                                                    </div>
                                                </FormControl>
                                                <FormMessage className="absolute left-0 -bottom-6" />
                                            </div>
                                        </FormItem>
                                    )}
                                />
                            </div>
                            
                            <div className="mt-1 mb-4">
                                <FormField 
                                    control={form.control}
                                    name="addresses"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center w-full flex-wrap gap-4">
                                            <div className="space-y-1 md:w-2/3">
                                            <FormLabel>{t('Sender IP Address(es)')}</FormLabel>
                                            <FormDescription>{t('IP addresses of devices sending flow data to this source.')}</FormDescription>
                                            </div>
                                            <FormControl className="flex-1">
                                                <Input {...field} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </div>
                            
                            <div className="mt-1 mb-4">
                                <FormField 
                                    control={form.control}
                                    name="port"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center w-full flex-wrap gap-4">
                                            <div className="space-y-1 md:w-2/3">
                                            <FormLabel>{t('Port')}</FormLabel>
                                            <FormDescription>
                                                {t('Unique port that flow data is being recieved on.')}
                                            </FormDescription>
                                            </div>
                                            <div className="relative flex-1 w-full">
                                                <FormControl className="flex-1">
                                                    <Input 
                                                        {...field}
                                                        value={field.value || ''}
                                                        onChange={(e) => field.onChange(Number(e.target.value))}
                                                    />
                                                </FormControl>
                                                <FormMessage className="absolute left-0 -bottom-6" />
                                            </div>
                                        </FormItem>
                                    )}
                                />
                            </div>
                            <div className="mt-1 mb-4">
                                <FormField 
                                    control={form.control}
                                    name="flowtype"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center w-full flex-wrap gap-4">
                                            <FormLabel className="text-sm font-medium md:w-2/3">{t('Flow Type')}</FormLabel>
                                            <FormControl className="flex-1">
                                                <div className="col-span-2 w-full">
                                                    <Select
                                                        disabled={!!source}
                                                        value={field.value}
                                                        onValueChange={(selectedVal) => {
                                                            field.onChange(selectedVal)
                                                        }}>
                                                        <SelectTrigger className="w-full">
                                                            <SelectValue />
                                                        </SelectTrigger>
                                                        <SelectContent>
                                                            <SelectGroup className="w-full max-h-80">
                                                                {flowTypes.map((flow, index) => (
                                                                    <SelectItem key={index} value={flow.toLowerCase()}>
                                                                        {flow}
                                                                    </SelectItem>
                                                                ))}
                                                            </SelectGroup>
                                                        </SelectContent>
                                                    </Select>
                                                </div>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </div>
                            <div className="mt-1 mb-4">
                                <FormItem className="flex items-center w-full flex-wrap gap-4">
                                    <div className="space-y-1 md:w-2/3">
                                        <FormLabel>
                                            {t('Raw Data Lifetime')}
                                        </FormLabel>
                                        <FormDescription>{t('Length of time you want granular flow data to be stored.')}</FormDescription>
                                    </div>
                                    <FormControl className="flex-1">
                                        <div className="flex items-center justify-between gap-4">
                                            <FormField
                                                control={form.control}
                                                name="lifetimeValue"
                                                render={({ field }) => (
                                                    <div className="relative flex-1 w-full">
                                                        <FormItem className="flex-1">
                                                            <FormControl>
                                                                <Input
                                                                    type="number"
                                                                    min={1}
                                                                    {...field}
                                                                />
                                                            </FormControl>
                                                        <FormMessage className="absolute left-0 -bottom-6" />
                                                        </FormItem >
                                                    </div>
                                                )}
                                            />
                                            <FormField
                                                control={form.control}
                                                name="lifetimeUnit"
                                                render={({ field }) => (
                                                    <Select
                                                        value={field.value}
                                                        onValueChange={field.onChange}
                                                    >
                                                        <SelectTrigger>
                                                            <SelectValue />
                                                        </SelectTrigger>
                                                        <SelectContent>
                                                            {lifetimeUnits.map(({ key, label }) => (
                                                                <SelectItem key={key} value={key}>
                                                                    {label}
                                                                </SelectItem>
                                                            ))}
                                                        </SelectContent>
                                                    </Select>
                                                )}
                                            />
                                        </div>
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            </div>
                            
                            <div className="mt-1">
                                <FormField 
                                    control={form.control}
                                    name="abnormal_behavior"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center w-full flex-wrap gap-4">
                                            <FormLabel className="text-sm font-medium md:w-2/3">{t('Abnormal Behavior Checks')}</FormLabel>
                                            <FormControl>
                                                <Checkbox
                                                    checked={!!field.value}
                                                    onCheckedChange={field.onChange}
                                                />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </div>
                            
                        </div>
                        <DialogFooter className="bg-card p-6">
                            <Button 
                                type="button" 
                                variant="secondary" 
                                onClick={() => handleOpenChange(false)}
                            >
                                {t("Cancel")}
                            </Button>
                            <Button type="submit">
                                {source ? t('Update Flow Source') : t('Create Flow Source')}
                            </Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
}