import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { ColumnDef } from "@tanstack/react-table";
import { Ellipsis, Pencil, Trash2, CirclePause, CirclePower, RotateCcw } from "lucide-react";
import { Source } from "@/components/source/types";
import { useTranslation } from "react-i18next";
import { useState } from "react";
import { SourceTraffic } from "@/components/source/SourceTraffic";
import { Link } from "react-router-dom";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import AddEditSourceDialog from "@/components/source/AddEditSourceDialog";
import { useAuth } from "@/contexts/AuthContext";
import {
  useDeleteSource,
  useStartSource,
  useStopSource,
  useRestartSource,
} from "@/components/source/queries";

type SourcesDataTableProps = {
  sources: Source[];
  isLoading?: boolean;
}

export default function SourcesDataTable({ sources, isLoading }: SourcesDataTableProps) {
  const { t } = useTranslation();
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [editingSource, setEditingSource] = useState<Source | undefined>(undefined);
  const { user } = useAuth();

  const startSourceMutation = useStartSource();
  const deleteSourceMutation = useDeleteSource();
  const stopSourceMutation = useStopSource();
  const restartSourceMutation = useRestartSource();

  const columns: ColumnDef<Source>[] = [
    {
      accessorKey: 'status',
      header: ({ column }) => <DataTableColumnHeader column={column} title="Status" />,
      cell: ({ row }) => (
        <div className="flex justify-left ml-5">
          <Tooltip>
            <TooltipTrigger asChild>
              <span
                className={`h-3 w-3 rounded-full inline-block ${
                  row.original.status ? "bg-success" : "bg-error"
                }`}
              />
            </TooltipTrigger>
            <TooltipContent>
              <p>{row.original.status ? t("Active") : t("Disabled")}</p>
            </TooltipContent>
          </Tooltip>
        </div>
      ),
    },
    {
      accessorKey: 'name',
      header: ({ column }) => <DataTableColumnHeader column={column} title="Flow Source Name" />,
      cell: ({ row }) => (
        <Link className="text-primary" to={`/sources/${row.original.id}`}>
          {row.original.name}
        </Link>
      ),
    },
    {
      accessorKey: 'traffic',
      id: 'traffic',
      header: t('Traffic Last 30 Minutes'),
      cell: ({ row }) => {
        const arr = row.original.traffic;
        let maxValue: number | null = null;

        if (Array.isArray(arr)) {
          const values = arr.map(a => a[0]).filter(v => v != null);
          if (values.length > 0) {
            maxValue = Math.max(...values);
          }
        }

        return Array.isArray(arr) && maxValue !== 0 && maxValue !== null
          ? <SourceTraffic data={arr} maxValue={maxValue} />
          : <span>{t('No Data')}</span>;
      },
    },
    {
      accessorKey: 'diskusage',
      header: ({ column }) => <DataTableColumnHeader column={column} title="Disk" />,
      cell: ({ row }) => <span>{row.original.diskusage || 'No Data'}</span>,
    },
    {
      accessorKey: 'lifetime',
      header: t('Data Lifetime'),
    },
    {
      accessorKey: 'flowtype',
      header: t('Flow Type'),
    },
    {
      accessorKey: 'port',
      header: ({ column }) => <DataTableColumnHeader column={column} title="Port" />,
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => (
        <>
          {!!user?.is_admin &&
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost">
                  <Ellipsis />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent>
                {!row.original.status && (
                  <DropdownMenuItem onClick={() => startSourceMutation.mutate(row.original.id!)}>
                    <CirclePower />
                    {t('Start')}
                  </DropdownMenuItem>
                )}
                {row.original.status && (
                  <DropdownMenuItem onClick={() => stopSourceMutation.mutate(row.original.id!)}>
                    <CirclePause />
                    {t('Stop')}
                  </DropdownMenuItem>
                )}
                {row.original.status && (
                  <DropdownMenuItem onClick={() => restartSourceMutation.mutate(row.original.id!)}>
                    <RotateCcw />
                    {t('Restart')}
                  </DropdownMenuItem>
                )}
                <DropdownMenuItem
                  onClick={() => {
                    setEditingSource(row.original);
                    setEditDialogOpen(true);
                  }}
                >
                  <Pencil />
                  {t('Edit')}
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => deleteSourceMutation.mutate(row.original.id!)}>
                  <Trash2 />
                  {t('Delete')}
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          }
        </>
      ),
    },
  ];

  return (
    <>
      {isLoading ? (
        <DataTableSkeleton></DataTableSkeleton>
      ) : (
        <DataTable columns={columns} data={sources} />
      )}
      {editingSource && (
        <AddEditSourceDialog
          source={editingSource}
          open={editDialogOpen}
          onOpenChange={(open) => {
            setEditDialogOpen(open);
            if (!open) setEditingSource(undefined);
          }}
        />
      )}
    </>
  );
}  