import { keepPreviousData, useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";
import { Source } from './types';
import { MultiSourceDataOptions } from '../dashboard/types';

export function useGetSources() {
    return useQuery<Source[], AxiosError>({
        queryKey: ['Sources'],
          queryFn: async () => {
              const response = await axios.get('/api/v1/source');
              return response.data;
          },
      });
}

export function useGetSource(id: number) {
    return useQuery<Source, AxiosError>({
      queryKey: ['Sources', id],
        queryFn: async () => {
            const response = await axios.get('/api/v1/source/' + id);
            return response.data;
        },
    });
}

export function useGetSourceSummaryChart(id: number, timeframe: string, props = {}) {
    return useQuery({
        queryKey: ['dashlet', 'source_summary', id, timeframe],
        queryFn: async () => {
            const response = await axios.get(`/api/v1/source/summary/chart/${id}`, {
                params: { timeframe }
            });
            return response.data;
        },
        ...props,
        placeholderData: keepPreviousData,
    });
}

export function useGetSourcesSummaryChart(timeframe: string, options: MultiSourceDataOptions, props = {}) {
    return useQuery({
        queryKey: ['dashlet', 'sources_summary', timeframe, options],
        queryFn: async () => {
            const response = await axios.get(`/api/v1/source/summary/chart`, {
                params: {
                    timeframe,
                    ...options,
                }
            });
            return response.data;
        },
        ...props,
        placeholderData: keepPreviousData,
    });
}

export function useGetSourceSummaryTalkers(id: number, timeframe: string) {
    return useQuery({
        queryKey: ['SummaryTalkers', id, timeframe],
        queryFn: async () => {
            const response = await axios.get(`/api/v1/source/summary/talkers/${id}`, {
                params: { timeframe }
            });
            return response.data;
        },
    });
}

export function useCreateSource() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (data: Source) => {
            return axios.post('/api/v1/source', data);
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['Sources'] });
            toast(t('Source successfully created.'));
        },
        onError: () => {
            toast(t('Failed to create source.'),
              { description: t('Please try again.') });
        },
    });
}

export function useEditSource() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (data: Source) => {
            return axios.put("/api/v1/source", data);
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["Sources"] });
            toast(t("Source has been edited."));
        },
        onError: () => {
            toast(t("Failed to edit source."), {
                description: t("Please try again."),
            });
        },
    });

}

export function useDeleteSource() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (id: number) => axios.delete(`/api/v1/source/${id}`),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['Sources'] });
            toast(t('Source deleted.'));
        },
        onError: () => {
            toast(t('Failed to delete source.'),
              { description: t('Please try again') });
        },
    });
}

export function useStartSource() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (id: number) => axios.post(`/api/v1/source/start/${id}`),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['Sources'] });
            toast(t('Source has successfully started.'));
        },
        onError: () => {
            toast(t('Failed to start source.'),
              { description: t('Please try again') });
        },
    });
}

export function useStopSource() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (id: number) => axios.post(`/api/v1/source/stop/${id}`),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['Sources'] });
            toast(t('Source has successfully stopped.'));
        },
        onError: () => {
            toast(t('Failed to stop source.'),
              { description: t('Please try again') });
        },
    });
}

export function useRestartSource() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (id: number) => axios.post(`/api/v1/source/restart/${id}`),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['Sources'] });
            toast(t('Source has successfully restarted.'));
        },
        onError: () => {
            toast(t('Failed to restart source.'),
              { description: t('Please try again') });
        },
    });
}
