import { useTranslation } from "react-i18next";
import { useForm } from "react-hook-form";
import { useState, useEffect } from "react";
import type { Source, QueryParam, QueryResults, FormParam } from "../types";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { columns } from "@/components/source/queries/columns";
import { Button } from "@/components/ui/button";
import { PageSeparator } from "@/components/Page";
import QueryDropdown from "@/components/source/queries/QueryDropdown";
import { useRunQuery } from "./queries";
import { Textarea } from "@/components/ui/textarea";
import { Input } from "@/components/ui/input";
import { AxiosError } from 'axios';
import {
    Card,
    CardContent,
    CardTitle,
    CardHeader,
    CardFooter,
} from "@/components/ui/card";
import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"
import { QuerySummaryTable, QuerySummarySkeleton } from "./QuerySummaryTable";
import { QueryChart, QueryChartSkeleton } from "./QueryChart";

export function Queries({src}: {src: Source[] | FormParam}){
    const { t } = useTranslation();

    const isFormParam = !Array.isArray(src);

    const [queryParams, setQueryParams] = useState<QueryParam | null>(isFormParam ? src : null);
    const { data, isLoading, isRefetching, refetch} = useRunQuery(queryParams);

    let formDefaults:FormParam;
    if (isFormParam){
        formDefaults = {
            ...src,
        }
    }
    else {
        formDefaults = {
            sources: src,
            raw_query: '',
            flow_start: '-2 hour',
            time_range: '2h',
        }
    }

    const form = useForm<FormParam>({
        defaultValues: formDefaults,
    });

    const handleRowClick = (row:QueryResults) => setQueryParams({
        sources: form.getValues('sources'),
        flow_start: form.getValues('flow_start'),
        flow_end: form.getValues('flow_end'),
        raw_query: `(${form.getValues('raw_query')}) and (src ip ${row.srcip} and dst ip ${row.dstip} and src port ${row.srcport} and dst port ${row.dstport})`,
    });

    const timeRange = form.watch("time_range");

    const presetTimeRanges: Record<
        Exclude<string, 'custom elapsed time' | 'custom date range'>,
        { flow_start: string; flow_end: string }> = {
            '2h': { flow_start: '-2 hour', flow_end: '-1 second' },
            '4h': { flow_start: '-4 hour', flow_end: '-1 second' },
            '6h': { flow_start: '-6 hour', flow_end: '-1 second' },
            '12h': { flow_start: '-12 hour', flow_end: '-1 second' },
            '24h': { flow_start: '-24 hour', flow_end: '-1 second' },
            '2d': { flow_start: '-2 day', flow_end: '-1 second' },
            '1w': { flow_start: '-1 week', flow_end: '-1 second' },
            '1m': { flow_start: '-1 month', flow_end: '-1 second' },
        };

    useEffect(() => {
        if (timeRange in presetTimeRanges) {
            const preset = presetTimeRanges[timeRange as keyof typeof presetTimeRanges];
            form.setValue('flow_start', preset.flow_start);
            form.setValue('flow_end', preset.flow_end);
        } else if (timeRange === 'custom elapsed time') {
            form.setValue('flow_end', '-1 second');
        }
        // else, if custom date range, do NOT override,
        // user fills these manually via inputs
        }, [timeRange]);

    useEffect(() => {
        if (queryParams) {
            form.setValue('raw_query', queryParams.raw_query);
            refetch()
                .then((result) => {
                    if (result.error) {
                        const axiosError = result.error as AxiosError<{ error: string }>;
                        const message = axiosError.response?.data?.error ??
                        t('Something has gone wrong with this query. The memory limit has likely been exceeded.')+
                        t(' Try searching a smaller timeframe, or using specific IPs or protocols. Check logs at')+
                        t(' /var/www/html/nagiosna/storage/logs/laravel.log for more information.');
                        form.setError('raw_query', { message });
                    }
                    })
                    .catch(() => {
                        form.setError('raw_query', { message: t('Unexpected error') });
                    });
        }
    }, [queryParams]);

    const onSubmit = async (formParam: FormParam) => {
        const { time_range, ...queryParam } = formParam;
        setQueryParams(queryParam);
    };

    return (
        <>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                <Form {...form}>
                    <form onSubmit={form.handleSubmit((onSubmit))}>
                        <Card className="h-full flex flex-col">
                            <CardContent>
                                <FormField
                                    control={form.control}
                                    name="time_range"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Time Frame')}</FormLabel>
                                                <FormDescription>{t('Set the time frame of the query')}</FormDescription>
                                            </div>
                                            <FormControl>
                                                <div className="w-[200px]">
                                                    <QueryDropdown field={field} />
                                                </div>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                                {( timeRange == 'custom elapsed time') &&
                                    <FormField
                                        control={form.control}
                                        name="flow_start"
                                        render={({ field }) => (
                                            <FormItem className="pt-2 flex items-center justify-between flex-wrap gap-4">
                                                <div className="space-y-1">
                                                    <FormDescription>{t('From: ')}</FormDescription>
                                                </div>
                                                <FormControl>
                                                    <div className="w-[200px]">
                                                        <Input placeholder={t("Flow Start")} {...field} value={field.value} />
                                                    </div>
                                                </FormControl>
                                                <FormMessage />
                                            </FormItem>
                                        )}
                                    />}
                                <PageSeparator />
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t('Raw Query')}</FormLabel>
                                        <FormDescription>
                                            {t("Uses nfdump raw query formating")}
                                        </FormDescription>
                                    </div>
                                    <div className="flex gap-2 w-md">
                                        <FormField
                                            control={form.control}
                                            name="raw_query"
                                            render={({ field }) => (
                                                <FormItem className="w-full">
                                                    <FormControl>
                                                        <Textarea {...field} className="col-span-3" />
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                    </div>
                                </FormItem>
                            </CardContent>
                            <CardFooter>
                                <Button type="submit">Run Query</Button>
                            </CardFooter>
                        </Card>
                    </form>
                </Form>
                {((data && !isRefetching) || isLoading || isRefetching) && <>
                    {data && !isRefetching && <QueryChart summary={data.summary} results={data.results} />}
                    {(isRefetching || isLoading) && <QueryChartSkeleton />}
                <Card>
                    <CardHeader>
                        <CardTitle>{t('Summary Data')}</CardTitle>
                    </CardHeader>
                    <CardContent>
                                {data && !isRefetching && <QuerySummaryTable {...data.summary} />}
                                {(isRefetching || isLoading) && <QuerySummarySkeleton />}
                    </CardContent>
                </Card>
                </>}
            </div>
            <div className="pt-8">
                {data && !isRefetching &&
                    <DataTable
                        columns={columns()}
                        data={ data.results }
                        onRowClick={handleRowClick}
                        sorting={[{ id: "Flow Start", desc: false }]}
                        usesRowSelection={false}
                        selectedRowActions={[]}
                        defaultColumnVisibility={{
                            'Flow End': false,
                            'Bits/Second': false,
                            'Packets/Second': false,
                            'Bytes/Packet': false,
                        }}
                    />}
                {(isLoading || isRefetching) && <DataTableSkeleton />}
            </div>
        </>
    )
}
