import React, { useState, useMemo, useEffect } from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useForm, SubmitHandler } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { useCreateSourceGroup, useEditSourceGroup } from "@/components/sourcegroups/queries";
import { SourceGroup } from "@/components/sourcegroups/types";
import { Source } from "@/components/source/types";
import { Textarea } from "@/components/ui/textarea";
import { useGetSources } from "@/components/source/queries";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";

export default function AddEditSourceDialog({ 
  sourcegroup, 
  children,
  open,
  onOpenChange
}: { 
  sourcegroup?: SourceGroup;
  children?: React.ReactNode;
  open?: boolean;
  onOpenChange?: (open: boolean) => void;
}) {
  const [internalOpen, setInternalOpen] = useState(false);
  const [selectedSources, setSelectedSources] = useState<Source[]>([]);
  const { t } = useTranslation();
  const createSourceGroup = useCreateSourceGroup();
  const updateSourceGroup = useEditSourceGroup();
  const { data } = useGetSources();

  // Use external state if provided, otherwise use internal state
  const isOpen = open !== undefined ? open : internalOpen;
  const handleOpenChange = onOpenChange || setInternalOpen;

  const unselectedSources = useMemo(() => {
    return (data || []).filter(
      (src: Source) => !selectedSources.find((s) => s.id === src.id)
    );
  }, [data, selectedSources]);

  const addSource = (source: Source) => {
    setSelectedSources((prev) => [...prev, source]);
  };

  const removeSource = (sourceId: number | undefined) => {
    setSelectedSources((prev) => prev.filter((s) => s.id !== sourceId));
  };

  const form = useForm<SourceGroup>({
    defaultValues: {
      id: sourcegroup?.id ?? undefined,
      name: sourcegroup?.name ?? "",
      description: sourcegroup?.description ?? "",
    },
  });

  const onSubmit: SubmitHandler<SourceGroup> = (data) => {
    const submitData: SourceGroup = {
      ...data,
      sources: selectedSources,
    };

    if (sourcegroup) {
      updateSourceGroup.mutate(submitData, {
        onSuccess: () => {
          handleOpenChange(false);
          form.reset();
        }
      });
    } else {
      createSourceGroup.mutate(submitData, {
        onSuccess: () => {
          handleOpenChange(false);
          form.reset();
        }
      });
    }
  };
    
  const handleDialogChange = (isOpen: boolean) => {
    handleOpenChange(isOpen);
    if (!isOpen) {
      form.reset();
    }
  }

  useEffect(() => {
    if (isOpen) {
      if (sourcegroup) {
        form.reset({
          id: sourcegroup.id,
          name: sourcegroup.name,
          description: sourcegroup.description,
        });
        setSelectedSources(sourcegroup.sources);
      } else {
        form.reset({
          name: "",
          description: "",
        });
        setSelectedSources([]);
      }
    }
  }, [isOpen, sourcegroup]);

  return (
    <Dialog open={isOpen} onOpenChange={handleDialogChange}>
      {children && (
        <DialogTrigger asChild>
          {children}
        </DialogTrigger>
      )}
      <DialogContent className="sm:max-w-[800px] p-0">
        <DialogHeader className="bg-card p-6">
          <DialogTitle>
            {sourcegroup ? t('Edit Flow Source Group') : t('Create Flow Source Group')}
          </DialogTitle>
          <DialogDescription>
            {t('You will need to add one or more source(s) to a sourcegroup. Fill out all information below.')}
          </DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <div className="flex flex-col gap-4 p-6">
              <FormField 
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem  className="flex items-center w-full flex-wrap gap-4">
                    <div className="space-y-1 md:w-2/3">
                      <FormLabel>{t('Flow Source Group Name')}</FormLabel>
                      <FormDescription>{t('Name of the flow source group, must be a unique name.')}</FormDescription>
                    </div>
                    <FormControl className="flex-1">
                      <div className="col-span-2 w-full">
                        <Input {...field} />
                      </div>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <div className="flex flex-col gap-4 p-6">
              <FormField 
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem  className="flex items-center w-full flex-wrap gap-4">
                    <div className="space-y-1 md:w-2/3">
                      <FormLabel>{t('Description')}</FormLabel>
                      <FormDescription>{t('Description of the flow source group.')}</FormDescription>
                    </div>
                    <FormControl className="flex-1">
                      <div className="col-span-2 w-full">
                        <Textarea {...field} />
                      </div>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <div className="flex flex-col gap-4 p-6">
              <FormLabel>{t("Sources")}</FormLabel>
              <FormDescription>{t("Select the sources you want in this flow source group.")}</FormDescription>
  
              <div className="flex flex-col md:flex-row gap-4">
                {/* Left - Available Sources */}
                <div className="w-full md:w-1/2 border rounded-lg p-4 bg-card">
                  <h4 className="font-semibold mb-2">{t("Available Sources")}</h4>
                  <ScrollArea className="h-60 rounded-md overflow-x-hidden">
                    <div className="space-y-2 p-1 pr-4">
                      {unselectedSources.map((source) => (
                        <div 
                          key={source.id} 
                          onClick={() => addSource(source)}
                          className="cursor-pointer bg-background hover:bg-black transition p-2 rounded shadow"
                        >
                          <Tooltip>
                            <TooltipTrigger asChild>
                              <span
                                className={`h-2 w-2 rounded-full inline-block mr-2 ${
                                  source.status ? "bg-success" : "bg-error"
                                }`}
                              />
                            </TooltipTrigger>
                            <TooltipContent>
                              <p>{source.status ? t("Active") : t("Disabled")}</p>
                            </TooltipContent>
                          </Tooltip>
                          {source.name}
                        </div>
                      ))}
                      {unselectedSources.length === 0 && (
                        <div className="text-sm text-muted-foreground">{t("No sources available.")}</div>
                      )}
                    </div>
                  </ScrollArea>
                </div>

                {/* Right - Selected Sources */}
                <div className="w-full md:w-1/2 border rounded-lg p-4 bg-card">
                  <h4 className="font-semibold mb-2">{t("Selected Sources")}</h4>
                  <ScrollArea className="h-60 rounded-md overflow-x-hidden">
                    <div className="space-y-2 p-1 pr-4">
                      {selectedSources.map((source) => (
                        <div
                          key={source.id}
                          onClick={() => removeSource(source.id)}
                          className="cursor-pointer bg-background hover:bg-black transition p-2 rounded shadow"
                        >
                          <Tooltip>
                            <TooltipTrigger asChild>
                              <span
                                className={`h-2 w-2 rounded-full inline-block mr-2 ${
                                  source.status ? "bg-success" : "bg-error"
                                }`}
                              />
                            </TooltipTrigger>
                            <TooltipContent>
                              <p>{source.status ? t("Active") : t("Disabled")}</p>
                            </TooltipContent>
                          </Tooltip>
                          {source.name}
                        </div>
                      ))}
                      {selectedSources.length === 0 && (
                        <div className="text-sm text-muted-foreground">{t("No sources selected.")}</div>
                      )}
                    </div>
                  </ScrollArea>
                </div>

              </div>
            </div>
            <DialogFooter className="bg-card p-6">
              <Button 
                type="button" 
                variant="secondary" 
                onClick={() => handleOpenChange(false)}
              >
                {t("Cancel")}
              </Button>
              <Button type="submit">
                {sourcegroup ? t('Update Flow Source Group') : t('Create Flow Source Group')}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}