import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";
import { SourceGroup } from "@/components/sourcegroups/types";
import type { Report, ReportDataItem } from '../report/types';

export function useGetSourceGroups() {
  return useQuery<SourceGroup[], AxiosError>({
    queryKey: ['sourceGroups'],
      queryFn: async () => {
        const response = await axios.get('/api/v1/sourcegroup');
        return response.data;
      },
  });
}

export function useGetSourceGroup(id: number) {
  return useQuery<SourceGroup, AxiosError>({
    queryKey: ['sourceGroups', id],
      queryFn: async () => {
          const response = await axios.get('/api/v1/sourcegroup/' + id);
          return response.data;
      },
  });
}

export function useCreateSourceGroup() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: (data: SourceGroup) => {
      return axios.post("/api/v1/sourcegroup", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["sourceGroups"] });
      toast(t('Flow source group successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create flow source group.'),
        { description: t('Please try again.') });
    }
  });
}

export function useEditSourceGroup() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: SourceGroup) => {
      return axios.put("/api/v1/sourcegroup", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["sourceGroups"] });
      toast(t("Flow source group has been edited."));
    },
    onError: () => {
      toast(t("Failed to edit flow source group."), {
        description: t("Please try again."),
      });
    },
  });
}

export function useDeleteSourceGroups() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (sourcegroupIds: number[]) => {
      return axios.delete(
        '/api/v1/sourcegroup',
        {
          data: { ids: sourcegroupIds },
        }
      );
    },
    onMutate: async (sourcegroupIds) => {
      const key = ['sourceGroups'];
      await queryClient.cancelQueries({ queryKey: key });
      const previousSourceGroups = queryClient.getQueryData<SourceGroup[]>(key);
      queryClient.setQueryData(key, (old: SourceGroup[] = []) =>
        old.filter((sourceGroup) => !sourcegroupIds.includes(sourceGroup.id))
      );
      return { previousSourceGroups };
    },
    onSuccess: (_data, sourcegroupIds) => {
      const message = sourcegroupIds.length > 1
        ? t('Flow source groups deleted.')
        : t('Flow source group deleted.');
      toast(message);
    },
    onError: (_error, sourcegroupIds, context) => {
      queryClient.setQueryData(['sourceGroups'], context?.previousSourceGroups)
      const errorMessage = sourcegroupIds.length > 1
        ? t('Failed to delete flow source groups.')
        : t('Failed to delete flow source group.');
      toast(errorMessage, {
        description: t('Please try again.'),
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['sourceGroups'] });
    },
  });
}

export function useGetSourceGroupSummaryChart(id: number, timeframe: string) {
  return useQuery({
      queryKey: ['SourceGroupSummaryChart', id, timeframe],
      queryFn: async () => {
          const response = await axios.get(`/api/v1/sourcegroup/summary/chart/${id}`, {
              params: { timeframe }
          });
          return response.data;
      },
  });
}

export function useGetSourceGroupSummaryTalkers(id: number, timeframe: string) {
  return useQuery({
      queryKey: ['SourceGroupSummaryTalkers', id, timeframe],
      queryFn: async () => {
          const response = await axios.get(`/api/v1/sourcegroup/summary/talkers/${id}`, {
              params: { timeframe }
          });
          return response.data;
      },
  });
}

export function useStartSourceGroup() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
      mutationFn: (id: number) => axios.post(`/api/v1/sourcegroup/start/${id}`),
      onSuccess: () => {
          queryClient.invalidateQueries({ queryKey: ['sourceGroups'] });
          toast(t('All sources in the flow source group were successfully started.'));
      },
      onError: () => {
          toast(t('Failed to start all sources in the flow source group.'),
            { description: t('Please try again') });
      },
  });
}

export function useStopSourceGroup() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
      mutationFn: (id: number) => axios.post(`/api/v1/sourcegroup/stop/${id}`),
      onSuccess: () => {
          queryClient.invalidateQueries({ queryKey: ['sourceGroups'] });
          toast(t('All sources in the flow source group were successfully stopped.'));
      },
      onError: () => {
          toast(t('Failed to stop all sources in the flow source group.'),
            { description: t('Please try again') });
      },
  });
}

export function useRestartSourceGroup() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
      mutationFn: (id: number) => axios.post(`/api/v1/sourcegroup/restart/${id}`),
      onSuccess: () => {
          queryClient.invalidateQueries({ queryKey: ['sourceGroups'] });
          toast(t('All sources in the flow source group were successfully restarted.'));
      },
      onError: () => {
          toast(t('Failed to restart all sources in the flow source group.'),
            { description: t('Please try again') });
      },
  });
}

export function useRunSourceGroupReport() {
  return useMutation({
    mutationFn: async ({ sourceGroupId, report }: { sourceGroupId: number, report: Report }) => {
      const response = await axios.post(`/api/v1/sourcegroup/${sourceGroupId}/reports/run`, report);
      return response.data;
    },
  });
}

export function useGetSourceGroupReportSankeyData() {
  return useMutation({
    mutationFn: async ({ sourceGroupId, report, reportData }: { 
      sourceGroupId: number, 
      report: Report, 
      reportData: ReportDataItem[] 
    }) => {
      const response = await axios.post(`/api/v1/sourcegroup/${sourceGroupId}/reports/sankeys`, {
        report,
        reportData
      });
      return response.data;
    },
  });
}