import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { useNavigate } from "react-router-dom";
import { useTranslation } from "react-i18next";
import { useState } from "react";

export interface SearchSuricataProps {
  srcIp?: string;
  destIp?: string;
  srcPort?: number;
  destPort?: number;
  protocol?: string;
  interface?: string;
  open: boolean;
  onOpenChange: (open: boolean) => void;
}

export default function SearchSuricata({
  srcIp,
  destIp,
  srcPort,
  destPort,
  protocol,
  interface: netInterface,
  open,
  onOpenChange,
}: SearchSuricataProps) {
  const { t } = useTranslation();
  const navigate = useNavigate();

  const fieldOptions = [
    { key: "srcIp", label: t("Source IP"), value: srcIp },
    { key: "destIp", label: t("Destination IP"), value: destIp },
    { key: "srcPort", label: t("Source Port"), value: srcPort },
    { key: "destPort", label: t("Destination Port"), value: destPort },
    { key: "protocol", label: t("Protocol"), value: protocol },
    { key: "interface", label: t("Interface"), value: netInterface },
  ];

  const validOptions = fieldOptions.filter(
    (field) => field.value !== undefined && field.value !== null && field.value !== ""
  );
  
  const [selectedFields, setSelectedFields] = useState<Record<string, boolean>>(
    Object.fromEntries(
      validOptions.map((f, _, arr) => [
        f.key,
        arr.length === 1 ? true : false,
      ])
    )
  );

  const handleToggleField = (key: string) => {
    setSelectedFields((prev) => ({ ...prev, [key]: !prev[key] }));
  };

  const handleSearch = () => {
    const filters: string[] = [];

    if (selectedFields.srcIp) filters.push(`"src_ip":"${srcIp}"`);
    if (selectedFields.destIp) filters.push(`"dest_ip":"${destIp}"`);
    if (selectedFields.srcPort) filters.push(`"src_port":${srcPort}`);
    if (selectedFields.destPort) filters.push(`"dest_port":${destPort}`);
    if (selectedFields.protocol) filters.push(`"proto":"${protocol}"`);
    if (selectedFields.interface) filters.push(`"interface":"${netInterface}"`);

    const filterString = filters.join(" && ");
    navigate(`/suricata?tab=data&filter=${encodeURIComponent(filterString)}`);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-lg">
        <DialogHeader>
          <DialogTitle className="text-lg font-semibold">{t("Search Suricata")}</DialogTitle>
        </DialogHeader>
        <p className="text-sm text-muted-foreground mb-3">
          {t("Select the filters you want to use for the Suricata search:")}
        </p>
        <div className="space-y-2">
          {validOptions.map((field) => (
            <div
              key={field.key}
              onClick={() => handleToggleField(field.key)}
              className={`flex items-center justify-between rounded-md px-3 py-2 shadow-sm border cursor-pointer transition ${
                selectedFields[field.key]
                  ? "bg-primary/10 border-primary"
                  : "bg-background hover:bg-muted"
              }`}
            >
              <div className="flex flex-col">
                <span className="text-sm font-medium">{field.label}</span>
                <span className="text-xs text-muted-foreground">{String(field.value)}</span>
              </div>
            </div>
          ))}
        </div>
        <DialogFooter className="mt-4">
          <Button variant="secondary" onClick={() => onOpenChange(false)}>
            {t("Cancel")}
          </Button>
          <Button 
            onClick={handleSearch} 
            disabled={!Object.values(selectedFields).some(Boolean)}
          >
            {t("Search")}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
