import { DataTable } from "@/components/ui/DataTable/DataTable";
import { useGetSuricataAlerts } from "@/components/suricata/alerts/queries";
import { alertColumns } from "@/components/suricata/alerts/alertColumns";
import {
    FormField,
    Form,
    FormLabel,
    FormItem,
    FormControl
} from "@/components/ui/form";
import {     
  Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue, 
  } from "@/components/ui/select";
import { useForm } from "react-hook-form";
import { useState, useEffect } from "react";
import { useTranslation } from "react-i18next";
import { type SortingState } from "@tanstack/react-table";
import { SuricataAlert } from "./types";
import { useNavigate } from "react-router";

export function SuricataAlerts({  alertFilter = "" }: { alertFilter: string }) {
  const { t } = useTranslation();
  const navigate = useNavigate();
  const form = useForm<{ time_period: string }>({
      defaultValues: { time_period: "last_week" },
  });
  const [selectedPeriod, setSelectedPeriod] = useState("last_week");
  const [debouncedPeriod, setDebouncedPeriod] = useState(selectedPeriod);
  const [ filter, setFilter ] = useState(alertFilter);

  useEffect(() => {
    const handler = setTimeout(() => setDebouncedPeriod(selectedPeriod), 100);
    return () => clearTimeout(handler);
  }, [selectedPeriod]);

  const { data, isLoading } = useGetSuricataAlerts(debouncedPeriod);

  const [sorting] = useState<SortingState>([
    { id: "count", desc: true },
  ]);

  const handleRowClick = (row: SuricataAlert) => { 
    navigate(`/suricata?tab=data&filter="signature":"${encodeURIComponent(row.signature)}"`);
  }

  const handleGlobalFilterChange = (value: string) => {
    setFilter(value);
  };

  return (
    <div className="space-y-2">
      <div className="flex justify-between items-center">
        <div className="flex justify-start">
          <Form {...form}>
            <form>
              <FormField
                control={form.control}
                name="time_period"
                render={({ field }) => (
                  <FormItem className="flex flex-col space-y-1">
                    <FormLabel className="whitespace-nowrap">{t("Time Period:")}</FormLabel>
                    <FormControl>
                      <Select
                        value={field.value}
                        onValueChange={(value) => {
                          field.onChange(value);
                          setSelectedPeriod(value);
                        }}
                      >
                        <SelectTrigger className="w-48">
                          <SelectValue placeholder="Select period" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="last_hour">{t("Last Hour")}</SelectItem>
                          <SelectItem value="last_day">{t("Last Day")}</SelectItem>
                          <SelectItem value="last_week">{t("Last Week")}</SelectItem>
                          <SelectItem value="last_month">{t("Last Month")}</SelectItem>
                          <SelectItem value="last_year">{t("Last Year")}</SelectItem>
                        </SelectContent>
                      </Select>
                    </FormControl>
                  </FormItem>
                )}
              />
            </form>
          </Form>
        </div>
      </div>
      <DataTable
        columns={alertColumns}
        data={data}
        isLoading={isLoading}
        sorting={sorting}
        onRowClick={handleRowClick}
        onGlobalFilterChange={handleGlobalFilterChange}
        defaultGlobalFilter={filter}
      />
    </div>
  );
}