import { useState } from "react";
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu"
import { Button } from "@/components/ui/button"
import { useTranslation } from "react-i18next";
import { 
  Ellipsis,
  Radar,
  ShieldQuestion,
  Waypoints,
} from "lucide-react";
import { DataActionPopup } from "@/components/suricata/data/DataActionPopup";
import AddNmapScanDialog from "@/components/nmap/scans/AddNmapScanDialog";
import SharkFin from "@/components/ui/icons/SharkFin";
import SearchRingBufferDialog from "@/components/wireshark/ringbuffer/SearchRingBufferDialog";
import { GetRingBuffer } from "@/components/wireshark/ringbuffer/queries";

export function DataActionDropdown({src, dest} : {src: string, dest: string}) {
  const { t } = useTranslation();
  const [popupOpen, setPopupOpen] = useState<null | "src" | "dest">(null);
  const [description, setDescription] = useState("");
  const [action, setAction] = useState<"whois" | "rdns">("whois");
  const [addNmapScanDialogOpen, setAddNmapScanDialogOpen] = useState(false);
  const [searchDialogOpen, setSearchDialogOpen] = useState(false);
  const [ip, setIP] = useState<string>("");
  const { data: ringbuffer } = GetRingBuffer();

  return (
    <>
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button
            variant="ghost"
            size="sm"
            className="ml-auto h-8 flex"
          >
            <Ellipsis />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="end">
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={e => {
              e.stopPropagation();
              setIP(src);
              setPopupOpen("src");
              setAction("whois");
              setDescription(t("Whois information for the source IP address"));
            }}
          >
            <ShieldQuestion />
            {t("Whois Src IP")}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={e => {
              e.stopPropagation();
              setIP(dest);
              setPopupOpen("dest");
              setAction("whois");
              setDescription(t("Whois information for the destination IP address"));
            }}
          >
            <ShieldQuestion />
            {t("Whois Dest IP")}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={e => {
              e.stopPropagation();
              setIP(src);
              setPopupOpen("src");
              setAction("rdns");
              setDescription(t("Reverse DNS information for the source IP address"));
            }}
          >
            <Waypoints />
            {t("Reverse DNS Src IP")}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={e => {
              e.stopPropagation();
              setIP(dest);
              setPopupOpen("dest");
              setAction("rdns");
              setDescription(t("Reverse DNS information for the destination IP address"));
            }}
          >
            <Waypoints />
            {t("Reverse DNS Dest IP")}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={e => {
              e.stopPropagation();
              setIP(src);
              setDescription(t("Nmap Scan Src IP"));
              setAddNmapScanDialogOpen(true);
            }}
          >
            <Radar />
            {t("Nmap Scan Src IP")}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={e => {
              e.stopPropagation();
              setIP(dest);
              setDescription(t("Nmap Scan Dest IP"));
              setAddNmapScanDialogOpen(true);
            }}
          >
            <Radar />
            {t("Nmap Scan Dest IP")}
          </DropdownMenuItem>
          {ringbuffer!== "none" && (
            <DropdownMenuItem
              className="cursor-pointer"
              onClick={e => {
                e.stopPropagation();
                setDescription(t("Find packets with IPs address"));
                setSearchDialogOpen(true);
              }}
            >
              <SharkFin />
              {t("Search Wireshark")}
            </DropdownMenuItem>
          )}
        </DropdownMenuContent>
      </DropdownMenu>
      <div onClick={e => {e.stopPropagation();}}>
        <AddNmapScanDialog
          open={addNmapScanDialogOpen}
          onOpenChange={setAddNmapScanDialogOpen}
          parameters={ip}
        >
          <span />
        </AddNmapScanDialog>

        <SearchRingBufferDialog
          open={searchDialogOpen}
          onOpenChange={setSearchDialogOpen}
          src={src}
          dst={dest}
        />
      </div>
      {popupOpen && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 cursor-default" onClick={e => {e.stopPropagation();}}>
            <DataActionPopup 
              description={description}
              ip={ip}
              action={action}
              setPopupOpen={setPopupOpen}
            />
        </div>
      )}
    </>
  );
}