import { allColumns } from "./dataColumns";
import { DataTable } from "../../ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { useEffect, useState } from "react";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/ui/select";
import { DetailSheet } from "@/components/ui/DetailSheet";
import {
    FormControl,
    FormField,
    FormItem,
    FormLabel,
    Form,
    FormMessage,
} from "@/components/ui/form";
import { logFileSelected, SuricataPcapScanUpdateEvent } from "@/components/suricata/data/types";
import {
    useGetConfig,
    useUpdateConfig,
} from "@/components/suricata/config/queries";
import {
    useGetFullJSONMutation,
    useGetJsonFiles,
    useGetSuricataData,
    useImportPcap,
} from "@/components/suricata/data/queries";
import { useTranslation } from "react-i18next";
import { useForm } from "react-hook-form";
import { SuricataConfigData } from "@/components/suricata/config/types";
import {
  useReactTable,
  getCoreRowModel,
  getSortedRowModel,
  getPaginationRowModel,
  type SortingState,
} from '@tanstack/react-table'
import { SuricataData } from "@/components/suricata/data/types";
import { Button } from "@/components/ui/button";
import { FileScan } from "lucide-react";
import ScanPcapDialog from "@/components/suricata/data/ScanPcapDialog";
import { useQueryClient } from "@tanstack/react-query";
import { useEcho } from "@laravel/echo-react";
import { toast } from "sonner";


export function SuricataDataTable({ initialFilter = "" }: { initialFilter?: string }) {
  const { t } = useTranslation();
  const [sheetOpen, setSheetOpen] = useState(false);
  const [sheetLoading, setSheetLoading] = useState(false);
  const [selectedRowData, setRowData] = useState<Record<
    string,
    unknown
  > | null>(null);
    const { data: jsonFileData, isLoading: isLoadingJsons} = useGetJsonFiles();
  const { data: configData } = useGetConfig();
  const updateConfig = useUpdateConfig();
  const [pageIndex, setPageIndex] = useState(0);
  const [pageSize, setPageSize] = useState(10);
  const [sorting, setSorting]= useState<SortingState>([{id:'', desc: false}]);
  const [filter,setFilter]= useState(initialFilter);
  const [pcapFileOpen, setPcapFileOpen] = useState(false);
  const { mutateAsync: importPcap } = useImportPcap();
  const queryClient = useQueryClient();
  const { data, isLoading } = useGetSuricataData(
    pageSize, 
    pageIndex + 1, 
    sorting[0]?.id || 'time', 
    sorting[0]?.desc ? 'desc' : 'asc',
    filter
  );

  useEffect(() => {
    setPageIndex(0);
  }, [sorting]);

  const logFileForm = useForm<logFileSelected>({
    values: {
      selectedFile: configData?.evelog_name ?? "eve.json",
    }
  })

  useEffect(() => {
  if (configData?.evelog_name) {
    logFileForm.setValue("selectedFile", configData.evelog_name);
    setSelectedFile(configData.evelog_name);
  }
  }, [configData?.evelog_name]);

  const [selectedFile, setSelectedFile] = useState(configData?.evelog_name ?? "")

  const { mutateAsync: fetchFullJSON } = useGetFullJSONMutation();

  const handleFileChange = async (data: SuricataConfigData) => {
    if (jsonFileData?.includes(data.evelog_name)) {
      await updateConfig.mutateAsync({ data, update: true });
    }
  }

  const handleGlobalFilterChange = (value: string) => {
    setPageIndex(0);
    setFilter(value);
  };

  const table = useReactTable<SuricataData['data'][number]>({
      data: data?.data ?? [],
      columns: allColumns,
      state: {
        sorting,
        pagination: { pageIndex, pageSize },
      },
      manualSorting: true,
      manualPagination: true,
      pageCount: data?.total
        ? Math.ceil(data.total / pageSize)
        : 0,
      onSortingChange: (updater) => {
        const next = typeof updater === 'function'
          ? updater(sorting)
          : updater
        setSorting(next)
      },
      onPaginationChange: (updater) => {
        const next = typeof updater === 'function'
          ? updater({ pageIndex, pageSize })
          : updater
        setPageIndex(next.pageIndex)
        setPageSize (next.pageSize)
      },
      getCoreRowModel:     getCoreRowModel(),
      getSortedRowModel:   getSortedRowModel(),
      getPaginationRowModel:getPaginationRowModel(),
    })

  const handleRowClick = async (row: { time: string; "flow ID": string }) => {
    setSheetOpen(true);
    setSheetLoading(true);

    const timestamp = encodeURIComponent(row.time);
    const flow_id = row["flow ID"];

    try {
      const data = await fetchFullJSON({ timestamp, flow_id });
      setRowData(data);
    } catch (error) {
      console.error("Failed to fetch full JSON:", error);
    } finally {
      setSheetLoading(false);
    }
  };

  useEcho(
  `NNA.Admins`,
  'SuricataPcapScanUpdate',
  (event: SuricataPcapScanUpdateEvent) => {
    if (event.status === 'success') {
      queryClient.invalidateQueries({ queryKey: ["AllSuricataData"] });
      queryClient.invalidateQueries({ queryKey: ["JsonFiles"] });
      queryClient.invalidateQueries({ queryKey: ["suricataConfig"] });
      toast.success("Suricata Scan Successful")
    }
    else {
      toast.error("Suricata Scan Failed")
    }
  }
);


  if (isLoadingJsons)
    return <DataTableSkeleton />;
  
  return (
    <div>
      <div className="flex items-center justify-between mb-2 gap-4">
        <div className="flex gap-4 flex-1">
          <div className="w-75">
            <Form {...logFileForm}>
              <form>
                <FormField
                  control={logFileForm.control}
                  name="selectedFile"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>
                        {t("Select log file:")}
                      </FormLabel>
                        <Select
                          onValueChange={(value) => {
                            setSelectedFile(value);
                            field.onChange(value);
                            configData!.evelog_name=value
                            handleFileChange(configData!)
                          }}
                          value={selectedFile}
                        >
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent className="max-h-100 scroll-y-auto">
                            {[
                              ...jsonFileData ?? []
                            ].map((option) => (
                              <SelectItem
                                key={option}
                                value={option}
                              >
                                {option}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                    </FormItem>
                  )}
                />
              </form>
            </Form>
          </div>
        </div>
        <Button
          onClick={() => {
              setPcapFileOpen(true);
          }}
        >
          <FileScan />
          {t("Scan Pcap")}
        </Button>
      </div>
      <DataTable
          table={table}
          onRowClick={handleRowClick}
          onGlobalFilterChange={handleGlobalFilterChange}
          defaultGlobalFilter={filter}
          isLoading={isLoading}
      />
      {sheetOpen && (
          <DetailSheet
              isOpen={sheetOpen}
              onOpenChange={setSheetOpen}
              JSONdata={selectedRowData ?? {}}
              title="Suricata Log Entry"
              description="Verbose details of the Suricata log."
              loading={sheetLoading}
          />
      )}
       {pcapFileOpen && (
          <ScanPcapDialog
            isOpen={pcapFileOpen}
            onOpenChange={setPcapFileOpen}
            onSubmit={async ({
                file,
                outputFileName,
            }) => {
              if (file) {
                try {
                  setPcapFileOpen(false);
                  await importPcap({
                    file: file,
                    output_file_name: outputFileName,
                  });
                } catch (error) {
                  console.error("Error processing PCAP:", error);
                }
              }
            }}
          />
        )}
    </div>
  )
}
