import axios from "axios";
import { useQuery, useMutation } from "@tanstack/react-query";
import { toast } from "sonner";
import {
  SuricataData,
  WhoisData,
  ReverseDNSData,
  ImportPcapParams,
} from "@/components/suricata/data/types";
import { useTranslation } from "react-i18next";

export function useGetSuricataData(per_page: number, page: number, sortBy: string, sortOrder: string, filter: string) {
  return useQuery<SuricataData>({
    queryKey: ["AllSuricataData", per_page, page, sortBy, sortOrder, filter],
    queryFn: async () => {
      const response = await axios.get<SuricataData>(
        "/api/v1/suricata/get",
        {
          params: {
            page,
            per_page,
            sortBy,
            sortOrder,
            filter,
          }
        }
      );
      return response.data;
    },
  });
}

export const fetchFullJSON = async (timestamp: string, flow_id: string) => {
  const res = await axios.get<Record<string, unknown>>(`/api/v1/suricata/fulljson?timestamp=${timestamp}&flow_id=${flow_id}`);
  return res.data;
};

export function useGetFullJSONMutation() {
  return useMutation({
    mutationFn: ({ timestamp, flow_id }: { timestamp: string; flow_id: string }) =>
      fetchFullJSON(timestamp, flow_id),
  });
}

export function useGetWhois(ip: string) {
  return useQuery<WhoisData>({
    queryKey: ["whois", ip],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/suricata/whois?ip=${ip}`);
      return response.data;
    },
    enabled: !!ip,
    retry: 1,
  });
}

export function useGetJsonFiles() {
  return useQuery<string[]>({
    queryKey: ["JsonFiles"],
    queryFn: async () => {
      const response = await axios.get("/api/v1/suricata/jsonfiles")
      return response.data
    }
  })
}

export function useGetReverseDNS(ip: string) {
  return useQuery<ReverseDNSData>({
    queryKey: ["reverseDNS", ip],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/suricata/reverseDNS?ip=${ip}`);
      return response.data;
    },
    enabled: !!ip,
  });
}

export function useImportPcap() {
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (data: ImportPcapParams) => {
      const formData = new FormData();
      formData.append("file", data.file as File);
      formData.append("output_file_name", data.output_file_name);
      return axios.post("/api/v1/suricata/import-pcap", formData, {
        headers: { "Content-Type": "multipart/form-data" },
      });
    },
    onMutate: () => {
      toast(t("Scanning PCAP file (this may take a moment) ..."));
    },
    retry: 3,
    retryDelay: attemptIndex => Math.min(1000 * attemptIndex, 5000), 
  });
}