import { Treemap } from "recharts";
import React from "react";
import { useTranslation } from "react-i18next";
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { useGetCategories } from "@/components/suricata/alerts/queries";
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/ui/chart";
import { useNavigate } from "react-router";
import { cn } from "@/lib/utils";

function getSeverityColor(category: string): string {
  const red    = "oklch(0.645 0.246 16.439)"; 
  const orange = "oklch(0.769 0.188 70.08)";   
  const green  = "oklch(0.696 0.17 162.48)";   
  const blue   = "oklch(0.60 0.25 260)";  

  // map categories to severity level
  const severityMap: Record<string, number> = {
    "Attempted User Privilege Gain": 1,
    "Unsuccessful User Privilege Gain": 1,
    "Successful User Privilege Gain": 1,
    "Attempted Administrator Privilege Gain": 1,
    "Successful Administrator Privilege Gain": 1,
    "Executable code was detected": 1,
    "A Network Trojan was detected": 1,
    "Web Application Attack": 1,
    "Inappropriate Content was Detected": 1,
    "Potential Corporate Privacy Violation": 1,
    "Targeted Malicious Activity was Detected": 1,
    "Exploit Kit Activity Detected": 1,
    "Domain Observed Used for C2 Detected": 1,
    "Successful Credential Theft Detected": 1,
    "Malware Command and Control Activity Detected": 1,

    "Potentially Bad Traffic": 2,
    "Attempted Information Leak": 2,
    "Information Leak": 2,
    "Large Scale Information Leak": 2,
    "Attempted Denial of Service": 2,
    "Denial of Service": 2,
    "Decode of an RPC Query": 2,
    "A suspicious filename was detected": 2,
    "An attempted login using a suspicious username was detected": 2,
    "A system call was detected": 2,
    "A client was using an unusual port": 2,
    "Detection of a Denial of Service Attack": 2,
    "Detection of a non-standard protocol or event": 2,
    "access to a potentially vulnerable web application": 2,
    "Misc Attack": 2,
    "Possible Social Engineering Attempted": 2,
    "Crypto Currency Mining Activity Detected": 2,
    "Possibly Unwanted Program Detected": 2,
    "Device Retrieving External IP Address Detected": 2,


    "Detection of a Network Scan": 3,
    "Unknown Traffic": 3,
    "Misc activity": 3,
    "A suspicious string was detected": 3,

    "Generic ICMP event": 4,
    "Generic Protocol Command Decode": 4,
    "A TCP connection was detected": 4,    
    "Not Suspicious Traffic": 4,
  };

  const level = severityMap[category] ?? 0;

  switch (level) {
    case 1: return red;
    case 2: return orange;
    case 3: return orange;
    case 4: return green;
    default: return blue; 
  }
}

type TreemapContentProps = {
  x: number;
  y: number;
  width: number;
  height: number;
  name: string;
  fill: string;
};

const CustomTreemapContent: React.FC<TreemapContentProps> = ({
  x,
  y,
  width,
  height,
  name,
  fill,
}) => {
  const navigate = useNavigate();
  const handleClick = () => {
    navigate(`/suricata?tab=alerts&alertFilter=${encodeURIComponent(name)}`)
  };

    if (width < 80 || height < 90) {
    return (
      <g onClick={handleClick} style={{ cursor: "pointer" }}>
        <rect x={x} y={y} width={width} height={height} fill={fill} stroke="#fff" />
      </g>
    );
  }

  const words = name?.split(" ");
  const lines: string[] = [];
  let currentLine = "";

  words?.forEach((word: string) => {
    if (width <= 110) {
        lines.push(currentLine.trim());
        currentLine = word;
      } else if (width > 110 && width <= 150 && (currentLine + " " + word).length > 20) {
        lines.push(currentLine.trim());
        currentLine = word;
      } else if (width > 150 && width <= 300 && (currentLine + " " + word).length > 25) {
        lines.push(currentLine.trim());
        currentLine = word;
      } else if (width > 300 && (currentLine + " " + word).length > 40) {
        lines.push(currentLine.trim());
        currentLine = word;
      } else {
        currentLine += " " + word;
      }
    });
  if (currentLine) lines.push(currentLine.trim());
  const fontSize = 12;
  const lineHeight = fontSize * 1.2; 
  return (
    <g onClick={handleClick} style={{ cursor: "pointer" }}>
      <rect x={x} y={y} width={width} height={height} fill={fill} stroke="#fff" />
      <text
        x={x + width / 2}
        textAnchor="middle"
        fontSize={fontSize}
        fontWeight="bold"
        fill="#fff"
      >
        {(() => {
          const centerY = y + height / 2;
          const firstBaselineY = centerY - ((lines.length - 1) * lineHeight) / 2;
          return lines.map((line, i) => (
            <tspan
              key={i}
              x={x + width / 2}
              y={firstBaselineY + i * lineHeight}
              dominantBaseline="middle"
            >
              {line}
            </tspan>
          ));
        })()}
      </text>
    </g>
  );
}

export function CategoryChart({ className }: { className?: string }) {
  const { t } = useTranslation();
  const { data: categoryData } = useGetCategories();
  const chartData = React.useMemo(() => {
    if (!categoryData) return [];
    return categoryData.map((item) => ({
      name: item.category,
      value: Number(item.count),
      fill: getSeverityColor(item.category),
    }));
  }, [categoryData]);

  const chartConfig: ChartConfig = React.useMemo(() => {
    if (!chartData.length) {
      return { value: { label: "Count" } } satisfies ChartConfig;
    }
    return {
      value: { label: "Count" },
      ...Object.fromEntries(
        chartData.map((item) => [
          item.name,
          { label: item.name, color: item.fill },
        ])
      ),
    } satisfies ChartConfig;
  }, [chartData]);

  return (
    <Card className={cn("w-full h-full", className)}>
      <CardHeader>
        <CardTitle>
          {t("Suricata Alert Categories From The Week")}
        </CardTitle>
      </CardHeader>
      <CardContent className="w-full h-full flex items-center justify-center overflow-hidden">
        {chartData.length === 0 ? (
          <div className="text-muted-foreground">{t("No category data available")}</div>
        ) : (
          <ChartContainer config={chartConfig} className="h-full w-full">
            <Treemap
              data={chartData}
              dataKey="value"
              nameKey="name"
              aspectRatio={4 / 3}
              animationDuration={0}
              content={
                ((props: TreemapContentProps) => (
                  <CustomTreemapContent {...props} />
                )) as unknown as React.ReactElement
              }
            >
              <ChartTooltip
                cursor={false}
                content={<ChartTooltipContent className="w-50" hideLabel={false} />}
              />
            </Treemap>
          </ChartContainer>
        )}
      </CardContent>
    </Card>
  );
}
