import {Sheet, SheetFooter, SheetContent, SheetHeader, SheetTitle, SheetDescription,} from "@/components/ui/sheet"
import { Form, FormLabel, FormControl, FormField, FormItem  } from '@/components/ui/form';
import { Input } from "@/components/ui/input";
import { SuricataRule } from "@/components/suricata/rules/types";
import { useForm } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { Select, SelectItem, SelectTrigger, SelectValue, SelectContent } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button"
import { SubmitHandler } from "react-hook-form";
import { useEditRule, useGetFullRule } from "@/components/suricata/rules/queries";
import { useEffect, useState } from "react";
import { Controller } from "react-hook-form";
import { Trash } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";

type AddEditRuleSheetProps = {
  open: boolean,
  setOpen: (open: boolean) => void;
  sid: string;
  title: string;
  description: string;
}


export function AddEditRuleSheet({ open, setOpen, sid, title, description }: AddEditRuleSheetProps) {
  const { t } = useTranslation();
  const editMutation = useEditRule();
  const { data } = useGetFullRule(sid, { enabled: open && !!sid && sid !== "-1" });
  const [addingOption, setAddingOption] = useState(false);
  const [newOptionKey, setNewOptionKey] = useState("");
  const [newOptionType, setNewOptionType] = useState<"string" | "boolean">("string");
  const form = useForm<SuricataRule>({
    defaultValues: {
      action: '',
      protocol: '',
      "source ip": '',
      "source port": '',
      direction: '',
      "destination ip": '',
      "destination port": '',
      status: '',
      options: {},
    },
  });

    // for editing rules
    useEffect(() => {
    if (open && data && sid !== "-1") {
      form.reset({
        action: data.action ?? "",
        protocol: data.protocol ?? "",
        "source ip": data["source ip"] ?? "",
        "source port": data["source port"] ?? "",
        direction: data.direction ?? "",
        "destination ip": data["destination ip"] ?? "",
        "destination port": data["destination port"] ?? "",
        status: data.status ?? "",
        options: data.options ?? {},
      });
      }
    }, [open, data, form]);

  // for new rules
  useEffect(() => {
    if (open && sid === "-1") {
      form.reset({
        action: "",
        protocol: "",
        "source ip": "",
        "source port": "",
        direction: "",
        "destination ip": "",
        "destination port": "",
        status: "active",
      });
      }
    }, [open, sid, form]);

    const onSubmit: SubmitHandler<SuricataRule> = (formData) => {
      editMutation.mutate({
        original_sid: sid,
        new_rule: formData,
      });
      setOpen(false);
    };

    const removeOption = (key: string) => {
      form.setValue(`options.${key}`, undefined);
    };

    // put msg and sid into the optional fields
    useEffect(() => {
      if (open && sid === "-1") {
        form.setValue("options.msg", "");
        form.setValue("options.sid", "");
      }
    }, [open, sid]);

  return (
    <Sheet open={open} onOpenChange={setOpen}>
      <SheetContent className="sm:max-w-[600px] flex flex-col">
        <SheetHeader className="mb-4">
          <SheetTitle>{title}</SheetTitle>
          <SheetDescription>
            {description}
          </SheetDescription>
        </SheetHeader>
          <ScrollArea className="overflow-y-hidden pr-4">
          <Form {...form}>
            <form id="edit-rule" onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-4">
              <FormField
                  control={form.control}
                  name="action"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0 pr-1">
                      <FormLabel className="text-right">{t('Action')}</FormLabel>
                        <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger className="w-full col-span-3">
                            <SelectValue placeholder="Select a rule action" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="alert">{t("alert")}</SelectItem>
                          <SelectItem value="pass">{t("pass")}</SelectItem>
                          <SelectItem value="drop">{t("drop")}</SelectItem>
                          <SelectItem value="reject">{t("reject")}</SelectItem>
                          <SelectItem value="rejectsrc">{t("rejectsrc")}</SelectItem>
                          <SelectItem value="rejectdst">{t("rejectdst")}</SelectItem>
                          <SelectItem value="rejectboth">{t("rejectboth")}</SelectItem>
                        </SelectContent>
                      </Select>
                    </FormItem>
                  )}
                />
              <FormField
                  control={form.control}
                  name="protocol"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0 pr-1">
                      <FormLabel className="text-right">{t('Protocol')}</FormLabel>
                      <FormControl>
                        <Input {...field} className="col-span-3" />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <div className="grid grid-cols-4 items-center gap-4">
                  <FormLabel className="text-right">{t('Source')}</FormLabel>
                  <div className="col-span-3 flex gap-4">
                    <FormField
                      control={form.control}
                      name="source ip"
                      render={({ field }) => (
                        <FormItem className="flex-1 space-y-0">
                          <FormControl>
                            <Input {...field} placeholder="IP Address" className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                    <span className="flex items-center -mt-1">:</span>
                    <FormField
                      control={form.control}
                      name="source port"
                      render={({ field }) => (
                        <FormItem className="w-18 space-y-0 pr-1">
                          <FormControl>
                            <Input {...field} placeholder="Port" className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <FormLabel className="text-right">{t('Destination')}</FormLabel>
                  <div className="col-span-3 flex gap-4">
                    <FormField
                      control={form.control}
                      name="destination ip"
                      render={({ field }) => (
                        <FormItem className="flex-1 space-y-0">
                          <FormControl>
                            <Input {...field} placeholder="IP Address" className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                    <span className="flex items-center -mt-1">:</span>
                    <FormField
                      control={form.control}
                      name="destination port"
                      render={({ field }) => (
                        <FormItem className="w-18 space-y-0 pr-1">
                          <FormControl>
                            <Input {...field} placeholder="Port" className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </div>
                </div>
                <FormField
                control={form.control}
                name="direction"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-4 items-center gap-4 pr-1">
                    <FormLabel className="text-right text-nowrap">{t('Direction')}</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger className="w-full col-span-3">
                            <SelectValue placeholder="Select a network direction" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="outbound">{t("Outbound (->)")}</SelectItem>
                          <SelectItem value="bidirectional">{t("Bidirectional (<>)")}</SelectItem>
                        </SelectContent>
                      </Select>
                  </FormItem>
                )}
              />
              <div className="py-5">Rule Options</div>
                {Object.entries(form.watch("options") ?? {}).map(([key, value]) => (
                  typeof value === "string" ? (
                    <Controller
                      key={key}
                      name={`options.${key}`}
                      control={form.control}
                      defaultValue={value}
                      render={({ field }) => (
                        <FormItem className="flex items-center gap-2 space-y-0 pr-1">
                          <FormLabel className="min-w-[120px]">{t(key)}</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              value={typeof field.value === "string" ? field.value : String(field.value ?? "")}
                              className="w-full"
                            />
                          </FormControl>
                          {key !== "msg" && key !== "sid" && (
                            <Button
                              type="button"
                              variant="ghost"
                              size="sm"
                              onClick={() => removeOption(key)}
                              title={t("Remove")}
                            >
                              <Trash />
                            </Button>
                          )}
                        </FormItem>
                      )}
                    />
                  ) : null
                ))}
                {Object.entries(form.watch("options") ?? {}).map(([key, value]) => (
                  typeof value === "boolean" ? (
                    <Controller
                      key={key}
                      name={`options.${key}`}
                      control={form.control}
                      defaultValue={value}
                      render={() => (
                        <FormItem className="grid grid-cols-6 items-center gap-4 space-y-0 pr-1">
                          <FormLabel className="text-right">{t(key)}</FormLabel>
                          <FormControl>
                            <div className="flex items-center gap-2">
                              <Button
                                type="button"
                                variant="ghost"
                                size="sm"
                                onClick={() => removeOption(key)}
                                className="ml-2"
                                title={t("Remove")}
                              >
                                <Trash />
                              </Button> 
                            </div>
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  ) : null
                ))}
                <FormField
                  control={form.control}
                  name="status"
                  render={({ field }) => (
                    <FormItem className="pt-2 grid grid-cols-6 items-center gap-4 space-y-0 pr-1">
                      <FormLabel className="text-right">{t('Enabled')}</FormLabel>
                      <FormControl>
                        <Checkbox 
                          checked={field.value === "active"}
                          onCheckedChange={(checked) => field.onChange(checked ? "active" : "inactive")} />
                      </FormControl>
                    </FormItem>
                  )}
                />
            </form>
          </Form>
          </ScrollArea>
      <SheetFooter className="flex flex-col sm:flex-row sm:items-end sm:justify-between gap-4">
        <div className="flex flex-wrap items-center gap-2">
          {!addingOption ? (
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => setAddingOption(true)}
            >
              +
            </Button>
          ) : (
            <>
              <Input
                value={newOptionKey}
                onChange={e => setNewOptionKey(e.target.value)}
                placeholder={t("Option name")}
                className="w-50"
                autoFocus
              />
              <Select
                value={newOptionType}
                onValueChange={v => setNewOptionType(v as "string" | "boolean")}
              >
                <SelectTrigger className="w-40">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="string">{t("With Argument")}</SelectItem>
                  <SelectItem value="boolean">{t("Without Argument")}</SelectItem>
                </SelectContent>
              </Select>
              <div className="justify-end">
              <Button
                type="button"
                size="sm"
                onClick={() => {
                  if (!newOptionKey) return;
                  form.setValue(
                    `options.${newOptionKey}`,
                    newOptionType === "boolean" ? true : ""
                  );
                  setAddingOption(false);
                  setNewOptionKey("");
                  setNewOptionType("string");
                }}
                variant="default"
              >
                {t("Add")}
              </Button>
              <Button
                type="button"
                size="sm"
                variant="ghost"
                onClick={() => {
                  setAddingOption(false);
                  setNewOptionKey("");
                  setNewOptionType("string");
                }}
              >
                {t("Cancel")}
              </Button>
              </div>
            </>
          )}
        </div>
        <Button form="edit-rule" type="submit" className="sm:w-auto w-full self-end">
          {t('Submit')}
        </Button>
      </SheetFooter>
      </SheetContent>
    </Sheet>
  );
}