import { DataTable } from "@/components/ui/DataTable/DataTable";
import { suricataRulesColumns } from "@/components/suricata/rules/rulesColumns";
import {
    useRuleDelete,
    useInfiniteSuricataRules,
    useGetRulesFiles,
} from "@/components/suricata/rules/queries";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import type { selectedRulesFile, SuricataRules as SuricataRulesType } from "@/components/suricata/rules/types";
import { DataTableRowAction } from "@/components/ui/DataTable/DataTable";
import { useRuleToggle, useImportRuleset } from "@/components/suricata/rules/queries";
import { AddEditRuleSheet } from "@/components/suricata/rules/AddEditRuleSheet";
import { Button } from "@/components/ui/button";
import { useState, useRef, useEffect } from "react";
import { Upload } from "lucide-react";
import { toast } from "sonner";
import { useTranslation } from "react-i18next";
import { useGetConfig, useUpdateConfig } from "@/components/suricata/config/queries";
import {
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  Form,
  FormMessage,
} from "@/components/ui/form";
import { useForm } from "react-hook-form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { SuricataConfigData } from "../config/types";
import { useAuth } from "@/contexts/AuthContext";

export function SuricataRules() {
  const { t } = useTranslation();
  const toggleMutation = useRuleToggle();
  const deleteMutation = useRuleDelete();
  const [isAddOpen, setAddOpen] = useState(false);
  const [allPagesLoaded, setAllPagesLoaded] = useState(false);
  const [selectedFile, setSelectedFile] = useState("")
  const importRuleSet = useImportRuleset();
  const updateConfig = useUpdateConfig();
  const { user } = useAuth();

  const {
    data: configData,
    isLoading: configLoading,
  } = useGetConfig();

  const form = useForm<selectedRulesFile>({
    values: {
      selectedFile: configData?.suricata_rules_name ?? "suricata.rules",
    }
  })

  useEffect(() => {
    if (configData?.evelog_name) {
      form.setValue("selectedFile", configData.suricata_rules_name);
      setSelectedFile(configData.suricata_rules_name)
    }
  }, [configData]);

  const { data: rulesFileData, isLoading: rulesFileisloading } = useGetRulesFiles()

  const pageSize = 25000;
  const {
    data,
    fetchNextPage,
    hasNextPage,
    isFetchingNextPage,
    isLoading: queryIsLoading,
    isError,
  } = useInfiniteSuricataRules(pageSize);
  const rules = data?.pages.flatMap((page) => page.rules) ?? [];

  useEffect(() => {
    if (!queryIsLoading && hasNextPage && !isFetchingNextPage) {
        fetchNextPage();
    }
  }, [
    rules.length,
    queryIsLoading,
    hasNextPage,
    isFetchingNextPage,
    fetchNextPage,
  ]);

  useEffect(() => {
    if (!hasNextPage && !isFetchingNextPage && data) {
      setAllPagesLoaded(true);
    }
  }, [hasNextPage, isFetchingNextPage, data]);

  const isLoading = queryIsLoading || (!allPagesLoaded && !isError);

  const actions: DataTableRowAction<SuricataRulesType["rules"][number]>[] = [
    {
      label: "Delete",
      value: "delete",
      onClick: (selectedRows) => {
        const sids = selectedRows.map((row) => row.SID);
        deleteMutation.mutate(sids);
      },
    },
    {
      label: "Turn on",
      value: "on",
      onClick: (selectedRows) => {
        const sids = selectedRows.map((row) => row.SID);
        toggleMutation.mutate({ sids, toggleMode: "on" });
      },
    },
    {
      label: "Turn off",
      value: "off",
      onClick: (selectedRows) => {
        const sids = selectedRows.map((row) => row.SID);
        toggleMutation.mutate({ sids, toggleMode: "off" });
      },
    },
  ];

  const fileInputRef = useRef<HTMLInputElement | null>(null);

  const uploadRuleset = () => {
    fileInputRef.current?.click();
  };

  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    if (!file.name.toLowerCase().endsWith(".rules") && !file.name.toLowerCase().endsWith(".txt")) {
      toast.error(t("Only .rules or .txt files are allowed"));
      return;
    }

    if (file.size > 50 * 1024 * 1024) {
      toast.error(t("File is too large (max 50MB)"));
      return;
    }
    const formData = new FormData();
    formData.append("ruleset", file);
    importRuleSet.mutate(formData);
  };

  const handleRuleFileChange = async (data: SuricataConfigData | undefined) => {
    if (!data) return; 
    await updateConfig.mutateAsync({ data, update: true });
  };

  if (isLoading || configLoading || rulesFileisloading) return (<DataTableSkeleton />)

  return (
    <div className="w-full flex flex-col">
      <div className="flex justify-between items-center mb-2">
        <div className="w-75 mb-2">
          <Form {...form}>
            <form>
              <FormField
                control={form.control}
                name="selectedFile"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>
                      {t("Select rule file:")}
                    </FormLabel>
                    <Select
                      onValueChange={(value) => {
                        setSelectedFile(value);
                        field.onChange(value);
                        if (configData) {
                          const updatedConfig = { ...configData, suricata_rules_name: value };
                          handleRuleFileChange(updatedConfig);
                        }
                      }}
                      value={selectedFile}
                    >
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent className="max-h-100 scroll-y-auto">
                        {[
                        ...rulesFileData ?? []
                        ].map((option) => (
                          <SelectItem
                            key={option}
                            value={option}
                          >
                            {option}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </form>
          </Form>
        </div>
        {!!user?.is_admin &&
          <div className="flex justify-end mb-2">
            <input
              type="file"
              accept=".rules,.txt"
              ref={fileInputRef}
              style={{ display: "none" }}
              onChange={handleFileChange}
            />
            <Button
              type="button"
              size="sm"
              onClick={() => uploadRuleset()}
              className="mr-2"
            >
              <Upload /> {t("Upload Ruleset")}
            </Button>
            <Button
              type="button"
              size="sm"
              onClick={() => setAddOpen(true)}
            >
              +
            </Button>
          </div>
          }
      </div>
      <DataTable
        columns={suricataRulesColumns}
        data={rules}
        usesRowSelection={true}
        selectedRowActions={actions}
      />
      <AddEditRuleSheet
        open={isAddOpen}
        setOpen={setAddOpen}
        sid={"-1"}
        title={t("Add Rule")}
        description={t("Add a new Suricata rule.")}
      />
    </div>
  );
}
