import axios from "axios";
import { useQuery, useQueryClient, useMutation, useInfiniteQuery } from "@tanstack/react-query";
import { toast } from "sonner";
import {
  ToggleRuleParams,
  SuricataRule,
  editRulesParam,
  SuricataData,
} from "@/components/suricata/rules/types";
import { useTranslation } from "react-i18next";

export function useRuleToggle() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (params: ToggleRuleParams) => {
      const sidsArray = Array.isArray(params.sids) ? params.sids : [params.sids];
      return axios.put("/api/v1/suricata/rules/toggle", {
        sids: sidsArray,
        toggleMode: params.toggleMode,
      });
    },

    onMutate: async (params) => {
      const sidsArray = Array.isArray(params.sids)
        ? params.sids.map(String)
        : [String(params.sids)];
      const sidSet = new Set(sidsArray);

      await queryClient.cancelQueries({ queryKey: ["suricataRules"] });

      const previousData = queryClient.getQueryData(["suricataRules"]);

      queryClient.setQueryData(["suricataRules"], (oldData: SuricataData) => {
        if (!oldData?.pages) return oldData;

        const newPages = oldData.pages.map((page) => ({
          ...page,
          rules: page.rules.map((rule) => {
            if (sidSet.has(String(rule.SID))) {
              const newStatus =
                params.toggleMode === "on"
                  ? "Active"
                  : params.toggleMode === "off"
                  ? "Inactive"
                  : rule.status === "Active"
                  ? "Inactive"
                  : "Active";

              return { ...rule, status: newStatus };
            }

            return rule;
          }),
        }));

        return {
          ...oldData,
          pages: newPages,
        };
      });

      return { previousData };
  },

    onError: (_err, _params, context) => {
      if (context?.previousData) {
        queryClient.setQueryData(["suricataRules"], context.previousData);
      }
      toast.error(t("Failed to toggle rules."));
    },

    onSuccess: () => {
      toast.success(t("Rules successfully toggled."));
    },

    onSettled: () => {
      debounce("invalidate-suricataRules", () => {
        queryClient.refetchQueries({
          queryKey: ["suricataRules"],
          type: "active",
        });
      }, 5000);
    },

    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * attemptIndex, 5000),
  });
}

export function useGetRulesFiles() {
  return useQuery<string[]>({
    queryKey: ["RulesFiles"],
    queryFn: async () => {
      const response = await axios.get("/api/v1/suricata/rules/rulesFiles")
      return response.data
    }
  })
}

export function useRuleDelete() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (sids: string | string[]) => {
      const sidsArray = Array.isArray(sids) ? sids : [sids];
      return axios.put("/api/v1/suricata/rules/delete", { sids: sidsArray });
    },

    onMutate: async (sids) => {
      toast(t("Deleting rules..."));

      const sidsArray = Array.isArray(sids) ? sids.map(String) : [String(sids)];
      const sidSet = new Set(sidsArray);

      await queryClient.cancelQueries({ queryKey: ["suricataRules"] });

      const previousData = queryClient.getQueryData(["suricataRules"]);

      queryClient.setQueryData(["suricataRules"], (oldData: SuricataData) => {
        if (!oldData?.pages) return oldData;

        const newPages = oldData.pages.map((page) => ({
          ...page,
          rules: page.rules.filter((rule) => !sidSet.has(String(rule.SID))),
        }));

        return {
          ...oldData,
          pages: newPages,
        };
      });

      return { previousData };
    },

    onError: (_err, _sids, context) => {
      if (context?.previousData) {
        queryClient.setQueryData(["suricataRules"], context.previousData);
      }
      toast.error(t("Failed to delete rules."));
    },

    onSuccess: () => {
      toast.success(t("Rules successfully deleted."));
    },

    onSettled: () => {
      debounce("invalidate-suricataRules", () => {
        queryClient.refetchQueries({
          queryKey: ["suricataRules"],
          type: "active",
        });
      }, 5000);
    },

    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * attemptIndex, 5000),
  });
}


export function useGetFullRule(sid: string, options?: { enabled?: boolean }) {
  return useQuery<SuricataRule>({
    queryKey: ["suricataFullRule", sid],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/suricata/rules/fullrule?sid=${sid}`);
      return response.data;
    },
    enabled: options?.enabled,
  });
}

export function useEditRule(){
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({original_sid, new_rule}: editRulesParam) => {
      return axios.put("/api/v1/suricata/rules/edit", {
        original_sid: original_sid,
        new_rule: new_rule
      });
    },
    onMutate: (variables) => {
      if (variables.original_sid === "-1") {
        toast(t("Adding rule..."));
      } else {
        toast(t("Editing rule..."));
      }
    },
    onSuccess: (_, variables) => {
      debounce("invalidate-suricataRules", () => {
        queryClient.refetchQueries({
        queryKey: ["suricataRules"],
        type: "active",
        });
      }, 1000);
      if (variables.original_sid === "-1") {
        toast(t("Rule successfully added."));
      } else {
        toast(t("Rule successfully edited."));
      }
    },
    onError: (_, variables) => {
      if (variables.original_sid === "-1") {
        toast(t("Failed to add rule."));
      } else {
        toast(t("Failed to edit rule."));
      }
    },
    retry: 3,
    retryDelay: attemptIndex => Math.min(1000 * attemptIndex, 5000), 
  });
}

export function useImportRuleset() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: FormData) => {
      return axios.post("/api/v1/suricata/rules/import", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["suricataRules"]})
      queryClient.invalidateQueries({ queryKey: ["suricataConfig"] });
      queryClient.invalidateQueries({ queryKey: ["RulesFiles"] });
      toast(t("Successfully uploaded ruleset."))
    },
    onError: () => {
      toast(t("Error uploading ruleset"))
    }
  })
}


export function useInfiniteSuricataRules(pageSize: number) {
  return useInfiniteQuery({
    queryKey: ["suricataRules"],
    queryFn: async ({ pageParam = 1 }) => {
      const res = await fetch(`/api/v1/suricata/rules?page=${pageParam}&pageSize=${pageSize}`);
      if (!res.ok) {
        throw new Error('Failed to fetch rules');
      }
      return res.json();
    },
    initialPageParam: 1,
    getNextPageParam: (lastPage, allPages) => {
      const totalPages = Math.ceil(lastPage.total / pageSize);
      const nextPage = allPages.length + 1;
      return nextPage <= totalPages ? nextPage : undefined;
    },
    retry: 1
  });
}

const timers: Record<string, NodeJS.Timeout> = {};
function debounce(key: string, fn: () => void, delay: number = 2000) {
  if (timers[key]) {
    clearTimeout(timers[key]);
  }
  timers[key] = setTimeout(() => {
    fn();
    delete timers[key];
  }, delay);
}