import { ColumnDef } from "@tanstack/react-table";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
  TooltipProvider,
} from "@/components/ui/tooltip";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { SuricataRules } from "@/components/suricata/rules/types";
import { useState, useRef, useEffect } from "react";
import { RulesDropdown } from "@/components/suricata/rules/RulesDropdown";
import { Checkbox } from "@/components/ui/checkbox";
import { useRuleToggle } from "@/components/suricata/rules/queries";
import { Button } from "@/components/ui/button"
import { useAuth } from "@/contexts/AuthContext";

const createColumn = <T extends SuricataRules["rules"][number]>(
  accessorKey: keyof T,
  title: string
): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => (
    <DataTableColumnHeader
      className="text-muted-foreground"
      column={column}
      title={title}
    />
  ),
  cell: ({ row }) => {
    const value = String(row.getValue(accessorKey as string));
    const textRef = useRef<HTMLDivElement>(null);
    const [isOverflowing, setIsOverflowing] = useState(false);

    useEffect(() => {
      const checkOverflow = () => {
        const el = textRef.current;
        if (el) {
          const isOver = el.scrollWidth > el.clientWidth;
          setIsOverflowing(isOver);
        }
      };

        let resizeTimeout: NodeJS.Timeout;
        const handleResize = () => {
          clearTimeout(resizeTimeout);
          resizeTimeout = setTimeout(() => {
            checkOverflow();
          }, 100);
        };

        checkOverflow();
        window.addEventListener("resize", handleResize);

        return () => {
          window.removeEventListener("resize", handleResize);
          clearTimeout(resizeTimeout);
        };
    }, [value]);

    const TextContent = (
      <div
        ref={textRef}
        className="pl-1 truncate whitespace-nowrap overflow-hidden text-ellipsis w-full max-w-[200px]"
      >
        {value}
      </div>
    );

    return isOverflowing ? (
      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger asChild>{TextContent}</TooltipTrigger>
          <TooltipContent>
            <p>{value}</p>
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    ) : (
      TextContent
    );
  },
});

const statusButton: ColumnDef<SuricataRules["rules"][number]> = {
  id: "status",
  accessorKey: "status",
  header: ({ column }) => (
    <DataTableColumnHeader
      className="text-muted-foreground"
      column={column}
      title={"Status"}
    />
  ),
  cell: ({ row }) => {
    const toggleMutation = useRuleToggle();
    const status = row.getValue("status") as "Active" | "Inactive";
    const handleClick = (e: React.MouseEvent<HTMLButtonElement>) => {
      e.preventDefault();
      toggleMutation.mutate({sids: row.original.SID, toggleMode: 'toggle'});
    };

    return (
      <Button
        variant= {status == "Active" ? "default" : "secondary"}
        onClick={handleClick} 
      >
        {status}
      </Button>
    );
  }
};

const actionsButton: ColumnDef<SuricataRules["rules"][number]> = {
  id: "actions",
  cell: ({ row }) => {
    const { user } = useAuth();
    return (
      <>
      {!!user?.is_admin &&
        <RulesDropdown 
          sid={row.original.SID}
        />
      }
      </>
    );
  }
};

const checkBox: ColumnDef<SuricataRules["rules"][number]> = {
  id: "select",
  header: ({ table }) => (
    <Checkbox
      checked={
        table.getIsAllPageRowsSelected() ||
        (table.getIsSomePageRowsSelected() && "indeterminate")
      }
      onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
      aria-label="Select all"
    />
  ),
  cell: ({ row }) => (
    <Checkbox
      checked={row.getIsSelected()}
      onCheckedChange={(value) => row.toggleSelected(!!value)}
      aria-label="Select row"
    />
  ),
  enableSorting: false,
  enableHiding: false,
}


export const suricataRulesColumns: ColumnDef<SuricataRules["rules"][number]>[] = 
  [
    checkBox,
    createColumn("SID", "SID"),
    createColumn("action", "Action"),
    createColumn("protocol", "Protocol"),
    createColumn("source ip", "Source IP"),
    createColumn("source port", "Source Port"),
    createColumn("direction", "Direction"),
    createColumn("destination ip", "Destination IP"),
    createColumn("destination port", "Destination Port"),
    statusButton,
    actionsButton
  ];