import {
  Form,
  FormItem,
  FormLabel,
  FormControl,
  FormMessage,
  FormField,
} from "@/components/ui/form"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogTitle,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { SuricataRuleset } from "@/components/suricata/rules/rulesets/types";
import { useForm } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { Select,
  SelectValue,
  SelectTrigger,
  SelectContent,
  SelectItem,
} from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { useEffect } from "react";
import { Input } from "@/components/ui/input";
import { useGetSuricataRuleset } from "@/components/suricata/rules/rulesets/queries";
import { PageSeparator } from "@/components/Page"
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { Info } from "lucide-react";

type AddEditRulesetDialogProps = {
  isOpen: boolean;
  onOpenChange: (isOpen: boolean) => void;
  onSubmit: (data: SuricataRuleset) => void;
  name: string;
}

export function AddEditRulesetDialog({
  isOpen,
  onOpenChange,
  onSubmit,
  name,
}: AddEditRulesetDialogProps) {
  const { t } = useTranslation();
  const addingRuleset = name === "";
  const { data: Ruleset, isLoading } = useGetSuricataRuleset(name, isOpen);
  const form = useForm<SuricataRuleset>();

  useEffect(() => {
    if (addingRuleset && isOpen) {
      form.reset({
        name: "",
        description: "",
        url: "",
        secretCode: "",
        secretCodeRequired: false,
        enabled: false,
        "Update Frequency": "",
        "Last Updated": "",
      });
    } else if (!isLoading && Ruleset) {
        form.reset({
          name: Ruleset.name || "",
          description: Ruleset.description || "",
          url: Ruleset.url || "",
          secretCode: Ruleset.secretCode || "",
          secretCodeRequired: Ruleset.secretCodeRequired || false,
          enabled: Ruleset.enabled || false,
          "Update Frequency": Ruleset["Update Frequency"] || "",
          "Last Updated": Ruleset["Last Updated"] || "",
        });
      }
  }, [Ruleset, isLoading, isOpen, addingRuleset, form]);

  const handleOpenChange = (open: boolean) => {
    if (!open) {
      form.reset(); 
    }
    onOpenChange(open);
  };

  const handleSubmit = (data: SuricataRuleset) => {
    onSubmit(data);
    onOpenChange(false);
  }
  const secretCodeRequired = form.watch("secretCodeRequired") ?? false;
  return (
    <Dialog open={isOpen} onOpenChange={handleOpenChange}>
      <DialogContent className="sm:max-w-[600px]">
          <DialogTitle>
            {addingRuleset ? t("Add Ruleset") : t("Edit Ruleset")}
          </DialogTitle>
          <DialogDescription>
            {addingRuleset ? t("Add a new ruleset here.") : t("Edit your rulesets here.")}
          </DialogDescription>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} id="edit-ruleset-form">
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                  <FormLabel>
                    {t("Name")}
                  </FormLabel>
                  <FormControl>
                    <Input {...field} disabled={!addingRuleset}/>
                </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="description"
              render={({ field }) => (
                <FormItem className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                  <FormLabel>
                    {t("Description")}
                  </FormLabel>
                  <FormControl>
                    <Input {...field} />
                </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="url"
              render={({ field }) => (
                <FormItem className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                  <FormLabel>
                    {t("URL")}
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <span tabIndex={0}><Info size={16} /></span>
                      </TooltipTrigger>
                      <TooltipContent>
                        {t("The URL of where to fetch the ruleset from")}
                      </TooltipContent>
                    </Tooltip>
                  </FormLabel>
                  <FormControl>
                    <Input {...field} />
                </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="secretCodeRequired"
              render={({ field }) => (
                <FormItem className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                  <FormLabel className="m-0">{t("Secret Code Required")}</FormLabel>
                  <FormControl>
                    <Checkbox
                      checked={!!field.value}
                      onCheckedChange={field.onChange}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="secretCode"
              render={({ field }) => (
                <FormItem className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                  <FormLabel>
                    {t("Secret Code")}
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <span tabIndex={0}><Info size={16} /></span>
                      </TooltipTrigger>
                      <TooltipContent>
                        {t("If the ruleset requires an authentication code enter it here")}
                      </TooltipContent>
                    </Tooltip>
                  </FormLabel>
                  <FormControl>
                    <Input {...field} disabled={!secretCodeRequired}/>
                </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="Update Frequency"
              render={({ field }) => (
                <FormItem className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                  <FormLabel>
                    {t("Update Frequency")}
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <span tabIndex={0}><Info size={16} /></span>
                      </TooltipTrigger>
                      <TooltipContent>
                        {t("How often suricata will try to update the ruleset")}
                      </TooltipContent>
                    </Tooltip>
                  </FormLabel>
                  <Select
                    onValueChange={field.onChange}
                    value={field.value}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder={t("Select frequency")} />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="5min">{t("Every 5 Minutes")}</SelectItem>
                      <SelectItem value="Hourly">{t("Hourly")}</SelectItem>
                      <SelectItem value="Daily">{t("Daily")}</SelectItem>
                      <SelectItem value="Weekly">{t("Weekly")}</SelectItem>
                      <SelectItem value="Monthly">{t("Monthly")}</SelectItem>
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="enabled"
              render={({ field }) => (
                <FormItem>
                  <div className="grid grid-cols-[175px_1fr] items-center gap-4 py-2">
                    <FormLabel className="m-0">{t("Enabled")}</FormLabel>
                    <FormControl>
                      <Checkbox
                        checked={!!field.value}
                        onCheckedChange={field.onChange}
                      />
                    </FormControl>
                  </div>
                  <FormMessage />
                </FormItem>
              )}
            />
          </form>
        </Form>
        <DialogFooter>
          <Button
            type="button"
            variant="secondary"
            onClick={() => onOpenChange(false)}
          >
            {t("Cancel")}
          </Button>
          <Button
            type="submit"
            form="edit-ruleset-form"
          >
            {t("Submit")}
            </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  )
}