import { DataTable} from "@/components/ui/DataTable/DataTable"
import { rulesetColumns } from "@/components/suricata/rules/rulesets/rulesetColumns"
import { AddEditRulesetDialog } from "@/components/suricata/rules/rulesets/AddEditRulesetDialog"
import { useEffect, useState } from "react"
import { useAddRuleset, useGetSuricataRulesets } from "@/components/suricata/rules/rulesets/queries"
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton"
import { Button } from "@/components/ui/button"
import { Info, Plus } from "lucide-react"
import { useTranslation } from "react-i18next"
import { SuricataRuleset } from "./types"
import type { DataTableRowAction } from "@/components/ui/DataTable/DataTable"
import { useDeleteRuleset, useToggleRuleset } from "@/components/suricata/rules/rulesets/queries"
import { Switch } from "@/components/ui/switch"
import { Card, CardContent } from "@/components/ui/card"
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip"
import { useGetConfig, useUpdateAutoRuleset } from "../../config/queries"
import { useAuth } from "@/contexts/AuthContext"

export function SuricataRulesets() {
  const { t } = useTranslation();
  const [isEditOpen, setEditOpen] = useState(false);
  const { mutate: addRulesetMutation } = useAddRuleset();
  const { data, isLoading } = useGetSuricataRulesets();
  const { mutate: deleteMutation } = useDeleteRuleset();
  const { mutate: toggleMutation } = useToggleRuleset();
  const { mutate: autoRuleset } = useUpdateAutoRuleset();
  const { data: configData } = useGetConfig();
  const { user } = useAuth();
  const [autoUpdate, setAutoUpdate] = useState<boolean>(false);
  const actions: DataTableRowAction<SuricataRuleset>[] = [
    {
      label: "Delete",
      value: "delete",
      onClick: (selectedRows) => {
        const names = selectedRows.map((row) => row.name);
        deleteMutation(names);
      },
    },
    {
      label: "Turn on",
      value: "on",
      onClick: (selectedRows) => {
        const names = selectedRows.map((row) => row.name);
        toggleMutation({ names, toggleMode: "on" });
      },
    },
    {
      label: "Turn off",
      value: "off",
      onClick: (selectedRows) => {
        const names = selectedRows.map((row) => row.name);
        toggleMutation({ names, toggleMode: "off" });
      },
    },
  ];

  useEffect(() => {
    if (configData?.auto_update_rules !== undefined) {
      setAutoUpdate(Boolean(configData.auto_update_rules));
      }
  }, [configData]);

  if (isLoading || !data) return (<DataTableSkeleton />)

  return (
    <div>
      {!!user?.is_admin &&
      <div className="pb-2 flex items-center justify-between gap-4">
        <Button
          onClick={() => {
            setEditOpen(true);
          }}
        >
          <Plus />
          {t("Add Ruleset")}
        </Button>
       
        <div className="flex items-center gap-2">
          <Card className="p-3">
            <CardContent className="flex gap-x-2 px-1 items-center">
              <Switch 
                id="auto-update" 
                checked={autoUpdate} 
                onCheckedChange={(checked) => {
                  autoRuleset(checked)
                  setAutoUpdate(checked)
                }}
              />
              <Tooltip>
                <TooltipTrigger asChild>
                  <span className="cursor-pointer">
                    <Info className="w-5 h-5" />
                  </span>
                </TooltipTrigger>
                <TooltipContent side="top" align="center">
                  {t("Automatically updates your Suricata rules file using the selected rulesets. WARNING: manual changes to suricata.rules will be overwritten.")}
                </TooltipContent>
              </Tooltip>
              <label className="font-bold text-sm text-muted-foreground">
                {t("Update Rulesets Automatically")}
              </label>
          </CardContent>
          </Card>
        </div>
      </div>
    }
      <DataTable
        columns={rulesetColumns}
        data={data}
        usesRowSelection={true}
        selectedRowActions={actions}
      />
      <AddEditRulesetDialog
        isOpen={isEditOpen}
        onOpenChange={setEditOpen}
        name = ""
        onSubmit={(data) => {
          addRulesetMutation(data)
          setEditOpen(false);
        }}
      />
    </div>
  )
}