import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { SuricataRuleset } from "./types";
import axios from "axios";
import { toast } from "sonner";


export function useGetSuricataRulesets() {
  return useQuery<SuricataRuleset[]>({
    queryKey: ['SuricataRulesets'],
    queryFn: async () => {
      const response = await axios.get<SuricataRuleset[]>("/api/v1/suricata/rulesets/getRulesets");
      return response.data;
    },
  })
}

export function useGetSuricataRuleset(name: string, enabled = true) {
  return useQuery<SuricataRuleset>({
    queryKey: ['SuricataRuleset', name],
    queryFn: async () => {
      const response = await axios.get<SuricataRuleset>(`/api/v1/suricata/rulesets/getRuleset?name=${name}`)
      return response.data;
    },
    enabled: !!name && enabled,
  })
}

export function useToggleRuleset() {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async ({
      names,
      toggleMode,
    }: {
      names: string[] | string;
      toggleMode: "toggle" | "on" | "off";
    }) => {
      return axios.put(`/api/v1/suricata/rulesets/toggle`, {
        names: Array.isArray(names) ? names : [names],
        toggleMode,
      });
    },
    onMutate: async ({ names, toggleMode }) => {
      await queryClient.cancelQueries({ queryKey: ["SuricataRulesets"] });

      const previousData = queryClient.getQueryData<SuricataRuleset[]>(["SuricataRulesets"]);

      queryClient.setQueryData(["SuricataRulesets"], (oldData: SuricataRuleset[] | undefined) => {
        if (!oldData) return oldData;

        const targetNames = Array.isArray(names) ? names : [names];

        return oldData.map((ruleset) => {
          if (!targetNames.includes(ruleset.name)) return ruleset;

          if (toggleMode === "toggle") {
            return { ...ruleset, enabled: !ruleset.enabled };
          } else if (toggleMode === "on") {
            return { ...ruleset, enabled: true };
          } else {
            return { ...ruleset, enabled: false };
          }
        });
      });

      return { previousData };
    },
    onError: (_err, _vars, context) => {
      if (context?.previousData) {
        queryClient.setQueryData(["SuricataRulesets"], context.previousData);
      }
      toast.error("Failed to update rulesets.");
    },
    onSuccess: () => {
      toast.success("Rulesets updated.");
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ["SuricataRulesets"] });
    },
  });
}

type useAddRulesetParams = {

    name: string;
    description: string | null;
    enabled: boolean;
    url: string;
    secretCode: string | null;
    secretCodeRequired: boolean;
    type: "Commercial" | "Open" | "Custom" | null;
    "Update Frequency": string;
    "Last Updated": string | null;
    "Fetch Status": string;
}

export function useAddRuleset() {
  const queryClient = useQueryClient();
  return useMutation ({
    mutationFn: (params: useAddRulesetParams) => {
      return axios.put('api/v1/suricata/rulesets/addRuleset', params)
    },
    onError: () => {
      toast.error("Failed to add ruleset.");
    },
    onSuccess: () => {
      toast.success("Ruleset added successfully.");
      queryClient.invalidateQueries({ queryKey: ["SuricataRulesets"] });
    },
  });
}

export function useEditRuleset() {
  const queryClient = useQueryClient();
  return useMutation ({
    mutationFn: (params: useAddRulesetParams) => {
      return axios.put('api/v1/suricata/rulesets/editRuleset', params)
    },
    onError: () => {
        toast.error("Failed to update ruleset.");
      },
      onSuccess: () => {
        queryClient.invalidateQueries({ queryKey: ["SuricataRulesets"] });
        toast.success("Ruleset updated.");
      },
  });
}

export function useDeleteRuleset() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (names: string | string[]) => {
      const payload = Array.isArray(names) ? { names } : { names: [names] };

      return axios.delete(`api/v1/suricata/rulesets/delete`, {
        data: payload, 
      });
    },
    onMutate: async (names: string | string[]) => {
      await queryClient.cancelQueries({ queryKey: ["SuricataRulesets"] });

      const previousRulesets = queryClient.getQueryData<SuricataRuleset[]>(["suricataRulesets"]);

      const nameArray = Array.isArray(names) ? names : [names];

      if (previousRulesets) {
        queryClient.setQueryData(
          ["suricataRulesets"],
          previousRulesets.filter(ruleset => !nameArray.includes(ruleset.name))
        );
      }

      return { previousRulesets };
    },
    onError: (_err, _names, context) => {
      if (context?.previousRulesets) {
        queryClient.setQueryData(["suricataRulesets"], context.previousRulesets);
      }
      toast.error("Failed to delete rulesets.")
    },
    onSuccess: () => {
      toast.success("Rulesets deleted.");
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ["SuricataRulesets"] });
    }
  });
}