import { Check, Clipboard } from "lucide-react";
import {
    Tooltip,
    TooltipContent,
    TooltipTrigger,
} from "@/components/ui/tooltip";
import { useTranslation } from "react-i18next";
import { Button } from "./button";
import { useEffect, useState } from "react";
import { toast } from "sonner";

type CopyButtonProps = {
    textToCopy: string;
};

export default function CopyButton({ textToCopy }: CopyButtonProps) {
    const { t } = useTranslation();
    const [isCopied, setIsCopied] = useState(false);

    const handleCopy = async () => {
        if (isCopied) return;
        if (!textToCopy) return;

        try {
            if (navigator?.clipboard?.writeText) {
                await navigator.clipboard.writeText(textToCopy);
            } else {
                const textarea = document.createElement("textarea");
                textarea.value = textToCopy;
                textarea.style.position = "fixed";
                textarea.style.opacity = "0";
                document.body.appendChild(textarea);
                textarea.select();
                document.execCommand("copy");
                document.body.removeChild(textarea);
            }

            setIsCopied(true);
        } catch (error) {
            console.log(error);
            toast.error(t("Failed to copy to clipboard."));
        }
    };

    useEffect(() => {
        if (isCopied) {
            const timer = setTimeout(() => {
                setIsCopied(false);
            }, 2000);
            return () => clearTimeout(timer);
        }
    }, [isCopied]);

    return (
        <Tooltip>
            <TooltipTrigger asChild>
                <Button
                    type="button"
                    variant="ghost"
                    className={`${isCopied && "cursor-default"} relative w-7 h-7`}
                    onClick={(e) => {
                        e.stopPropagation();
                        handleCopy();
                    }}
                >
                    <Clipboard
                        size={16}
                        className={`absolute transition-all duration-200 ${
                            isCopied ? "scale-0" : "scale-100"
                        }`}
                    />
                    <Check
                        size={16}
                        className={`absolute transition-all duration-200 ${
                            isCopied ? "scale-100" : "scale-0"
                        }`}
                    />
                    <span className="sr-only">
                        {isCopied ? t("Copied") : t("Copy to Clipboard")}
                    </span>
                </Button>
            </TooltipTrigger>
            <TooltipContent>
                <p>{isCopied ? t("Copied") : t("Copy to Clipboard")}</p>
            </TooltipContent>
        </Tooltip>
    );
}
