import * as React from "react";
import { DataTableToolbar } from "./DataTableToolbar";
import { DataTableRowSelection } from "./DataTableRowSelection";
import { useTranslation } from "react-i18next";
import {
  ColumnDef,
  getFilteredRowModel,
  flexRender,
  getCoreRowModel,
  useReactTable,
  getPaginationRowModel,
  getSortedRowModel,
  type Table as TanstackTable,
  type SortingState,
  type VisibilityState,
} from "@tanstack/react-table";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { DataTablePaginationNav } from "@/components/ui/DataTable/DataTablePaginationNav";
interface DataTableProps<TData, TValue> {
  columns?: ColumnDef<TData, TValue>[];
  data?: TData[];
  table?: TanstackTable<TData>;
  sorting?: SortingState;
  onRowClick?: (row: TData) => void;
  checkActiveRow?: (row: TData) => boolean;
  usesRowSelection?: boolean;
  selectedRowActions?: DataTableRowAction<TData>[];
  onGlobalFilterChange?: (value: string) => void;
  defaultColumnVisibility?: VisibilityState;
  isLoading?: boolean;
  hideView?: boolean;
  hidePageSizeSelector?: boolean;
  hidePagination?: boolean;
  hideSearch?: boolean;
  defaultGlobalFilter?: string;
}
export interface DataTableRowAction<TData> {
  label: string;
  value: string;
  onClick: (selectedRows: TData[]) => void | Promise<void>;
}
export function DataTable<TData, TValue>({
  columns,
  data,
  table: externalTable,
  sorting: sortingProp,
  onRowClick,
  checkActiveRow,
  usesRowSelection,
  selectedRowActions,
  defaultColumnVisibility,
  onGlobalFilterChange,
  isLoading,
  hideView,
  hidePageSizeSelector,
  hidePagination,
  hideSearch,
  defaultGlobalFilter = "",
}: DataTableProps<TData, TValue>) {
  const [sorting, setSorting] = React.useState<SortingState>(sortingProp ?? []);
  const [columnVisibility, setColumnVisibility] = React.useState<VisibilityState>(
    defaultColumnVisibility ?? {}
  );
  const [globalFilter, setGlobalFilter] = React.useState(defaultGlobalFilter);
  const [rowSelection, setRowSelection] = React.useState({});
  const { t } = useTranslation();
  const internalTable = useReactTable({
    data: data ?? [],
    columns: columns ?? [],
    getCoreRowModel: getCoreRowModel(),
    getPaginationRowModel: getPaginationRowModel(),
    onSortingChange: setSorting,
    getSortedRowModel: getSortedRowModel(),
    onColumnVisibilityChange: setColumnVisibility,
    getFilteredRowModel: getFilteredRowModel(),
    onGlobalFilterChange: setGlobalFilter,
    globalFilterFn: "includesString",
    onRowSelectionChange: setRowSelection,
    state: {
      sorting,
      columnVisibility,
      globalFilter,
      rowSelection,
    },
    autoResetPageIndex: false,
  });
  const table = externalTable ?? internalTable;
  return (
    <div className="flex flex-col gap-2">
      <DataTableToolbar table={table} onGlobalFilterChange={onGlobalFilterChange} hideView={hideView} hidePageSizeSelector={hidePageSizeSelector} hidePagination={hidePagination} hideSearch={hideSearch} globalFilter={globalFilter}/>
      <div className="rounded-md border overflow-x-auto">
        <Table>
          <TableHeader>
            {table.getHeaderGroups().map(headerGroup => (
              <TableRow key={headerGroup.id}>
                {headerGroup.headers.map(header => (
                  <TableHead key={header.id} colSpan={header.colSpan}>
                    {header.isPlaceholder
                      ? null
                      : flexRender(header.column.columnDef.header, header.getContext())}
                  </TableHead>
                ))}
              </TableRow>
            ))}
          </TableHeader>
          <TableBody>
            {isLoading ? (
              Array.from({ length: 10 }).map((_, idx) => (
                <TableRow key={`skeleton-${idx}`}>
                  {table.getVisibleFlatColumns().map((column, colIdx) => (
                    <TableCell key={column.id ?? colIdx}>
                      <div className="h-4 bg-muted rounded animate-pulse w-full" />
                    </TableCell>
                  ))}
                </TableRow>
              ))
            ) : table.getRowModel().rows.length ? (
              table.getRowModel().rows.map(row => (
                  <TableRow
                    key={row.id}
                    data-state={row.getIsSelected() ? "selected" : undefined}
                    onClick={onRowClick ? () => onRowClick(row.original) : undefined}
                    className={`cursor-pointer transition-colors duration-200 hover:bg-muted/50 ${
                      checkActiveRow && checkActiveRow(row.original)
                        ? "bg-secondary font-semibold"
                        : ""
                    }`}
                  >
                    {row.getVisibleCells().map(cell => (
                      <TableCell key={cell.id} className="overflow-hidden">
                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                      </TableCell>
                    ))}
                  </TableRow>
                )
              )
            ) : (
              <TableRow>
                <TableCell
                  colSpan={table.getAllColumns().length}
                  className="h-24 text-center"
                >
                  {t("No results.")}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
      <div className="flex items-center justify-between">
        <div className="flex-1">
          {usesRowSelection && selectedRowActions && (
            <DataTableRowSelection table={table} actions={selectedRowActions} />
          )}
        </div>
        <div className="flex justify-end px-2">
          {!hidePagination && <DataTablePaginationNav table={table} />}
        </div>
      </div>
    </div>
  );
}
