import { Table } from "@tanstack/react-table";
import { DataTableViewOptions } from "./DataTableViewOptions";
import { Input } from "@/components/ui/input";
import { DataTablePagination } from "./DataTablePagination";
import { Button } from "@/components/ui/button";
import { useState } from "react";
import { useTranslation } from "react-i18next";

interface DataTableToolbarProps<TData> {
    table: Table<TData>;
    onGlobalFilterChange?: (value: string) => void;
    hideView?: boolean;
    hidePageSizeSelector?: boolean;
    hidePagination?: boolean;
    hideSearch?: boolean;
    globalFilter?: string;
}

export function DataTableToolbar<TData>({
    table,
    onGlobalFilterChange,
    hideView=false,
    hidePageSizeSelector=false,
    hidePagination=false,
    hideSearch=false,
    globalFilter,
}: DataTableToolbarProps<TData>) {
    const [inputValue, setInputValue] = useState(globalFilter ?? table.getState().globalFilter ?? "");
    const { t } = useTranslation();
    const handleSearch = () => {
        table.setGlobalFilter(inputValue);
        onGlobalFilterChange?.(inputValue);
    };
    const handleKeyDown = (e: React.KeyboardEvent<HTMLInputElement>) => {
        if (e.key === "Enter") {
            handleSearch();
        }
    };
    return (
        <div className="flex flex-row items-center justify-between">
            <div className="flex flex-1 items-center space-x-2">
                {!hideSearch &&
                    <>
                        <Input
                            placeholder="Search..."
                            value={inputValue}
                            onChange={(e) => {
                                const val = e.target.value;
                                setInputValue(val);
                                if (val.trim() === "") {
                                    table.setGlobalFilter("");
                                    onGlobalFilterChange?.("");
                                }
                            }}
                            onKeyDown={handleKeyDown}
                            className="max-w-60 focus:ring-inset h-8"
                        />
                        <Button type="button" onClick={handleSearch}  variant="outline" size="sm" className="h-8 lg:flex">{t("Search")}</Button>
                    </>
                }
            </div>
            {
                !hideView && <DataTableViewOptions table={table} />
            }
            {!hidePagination && <DataTablePagination table={table} hidePageSizeSelector={hidePageSizeSelector} />}
        </div>
    );
}
