import { useEffect, useState } from "react";
import { LoaderCircle } from "lucide-react";

type DurationProps = {
  started_at?: string | null;
  finished_at?: string | null;
};

export function Duration({ started_at, finished_at }: DurationProps) {
  const startTime = parseDate(started_at);
  const endTime = parseDate(finished_at);

  const [now, setNow] = useState<Date>(new Date());

  useEffect(() => {
    if (!startTime || endTime) return;

    const interval = setInterval(() => {
      setNow(new Date());
    }, 1000);

    return () => clearInterval(interval);
  }, [startTime, endTime]);

  const formatDuration = (ms: number) => {
    const seconds = Math.floor((ms / 1000) % 60);
    const minutes = Math.floor((ms / (1000 * 60)) % 60);
    const hours = Math.floor(ms / (1000 * 60 * 60));
    return [hours ? `${hours}h` : null, minutes ? `${minutes}m` : null, `${seconds}s`]
      .filter(Boolean)
      .join(" ");
  };

  if (!startTime) return <div />;

  const durationMs = endTime
    ? endTime.getTime() - startTime.getTime()
    : now.getTime() - startTime.getTime();

  return (
    <span className="flex items-center">
      {formatDuration(Math.max(durationMs, 0))}
      {!endTime && <LoaderCircle className="animate-spin ml-2 h-4 w-4" />}
    </span>
  );
}

function parseDate(value?: string | null): Date | null {
  if (!value) return null;

  if (/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(?:\.\d+)?([Z]|[+-]\d{2}:\d{2})$/.test(value)) {
    return new Date(value);
  }
  return new Date(value.replace(" ", "T") + "Z");
}
