import React, { useState, useEffect } from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  Card,
  CardContent,
} from "@/components/ui/card";
import { RadioGroup } from '@radix-ui/react-radio-group';
import { RadioGroupItem } from '@/components/ui/radio-group';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from '@/components/ui/label';
import { PasswordInput } from '@/components/ui/passwordInput';
import ThemesDropdown from "@/components/ui/ThemesDropdown";
import { useForm, SubmitHandler } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { useCreateUser, useEditUser } from "@/components/user/queries";
import { useGetAuthServers } from "@/components/administration/queries";
import type { Language, User } from "@/components/user/types";
import type { AuthServer } from "@/components/administration/types";
import { cn } from "@/lib/utils";
import { Check, ChevronsUpDown } from "lucide-react";
import { PageSeparator } from "@/components/Page";

export default function AddEditUserDialog({ 
    user, 
    children,
    open,
    onOpenChange
}: { 
    user?: User;
    children?: React.ReactNode;
    open?: boolean;
    onOpenChange?: (open: boolean) => void;
}) {
    const [internalOpen, setInternalOpen] = useState(false);
    const [languageOpen, setLanguageOpen] = useState(false);
    const [authType, setAuthType] = useState<'Local' | 'LDAP' | 'AD'>('Local');
    const { t } = useTranslation();
    const createUser = useCreateUser();
    const updateUser = useEditUser();
    const { data: authServers } = useGetAuthServers();

    // Use external state if provided, otherwise use internal state
    const isOpen = open !== undefined ? open : internalOpen;
    const handleOpenChange = onOpenChange || setInternalOpen;

    const ldapServers = (authServers ?? []).filter((server: AuthServer) => server.type === "ldap");
    const adServers = (authServers ?? []).filter((server: AuthServer) => server.type === "ad");

    const languages: { label: string; value: Language }[] = [
        { label: t("Bulgarian"), value: "bg_BG" },
        { label: t("Czech"), value: "cs_CZ" },
        { label: t("German"), value: "de_DE" },
        { label: t("English"), value: "en_US" },
        { label: t("Spanish"), value: "es_ES" },
        { label: t("French"), value: "fr_FR" },
        { label: t("Italian"), value: "it_IT" },
        { label: t("Japanese"), value: "ja_JP" },
        { label: t("Korean"), value: "ko_KR" },
        { label: t("Polish"), value: "pl_PL" },
        { label: t("Portugese"), value: "pt_PT" },
        { label: t("Russian"), value: "ru_RU" },
        { label: t("Chinese (China)"), value: "zh_CN" },
        { label: t("Chinese (Taiwan)"), value: "zh_TW" },
    ];

    const form = useForm<User & { confirm_password: string }>({
        defaultValues: {
            id: user?.id ?? 0,
            type: user?.type ?? "Local",
            first_name: user?.first_name ?? "",
            last_name: user?.last_name ?? "",
            company: user?.company ?? "",
            phone: user?.phone ?? "",
            username: user?.username ?? "",
            password: "",
            confirm_password: "",
            email: user?.email ?? "",
            lang: user?.lang ?? "en_US",
            theme: user?.theme ?? "Default",
            is_admin: user?.is_admin ? "1" : "0",
            apiaccess: user?.apiaccess ? "1" : "0",
            default_dashboard: user?.default_dashboard ?? 0,
            auth_server_id: user?.auth_server_id ?? 0,
            auth_server_data: user?.auth_server_data ?? "",
        }
    });

    // Set initial auth type based on user data
    useEffect(() => {
        if (user?.type) {
            setAuthType(user.type as 'Local' | 'LDAP' | 'AD');
        }
    }, [user]);

    useEffect(() => {
      if (!authServers) return;
      
      // Only set default auth server for new users or when user doesn't have an existing auth_server_id
      if (!user || !user.auth_server_id) {
          if (authType == "AD" && adServers.length > 0) {
              form.setValue("auth_server_id", adServers[0].id, { shouldDirty: true });
          } else if (authType == "LDAP" && ldapServers.length > 0) {
              form.setValue("auth_server_id", ldapServers[0].id, { shouldDirty: true });
          }
      }
  }, [authServers, authType, adServers, ldapServers, user]);

    const onSubmit: SubmitHandler<User & { confirm_password: string }> = (data) => {
      const submitData: User = {
          ...data,
          is_admin: data.is_admin,
          apiaccess: data.apiaccess,
      };

      if (user) {
          updateUser.mutate({ data: submitData }, {
              onSuccess: () => {
                  handleOpenChange(false);
                  form.reset();
              }
          });
      } else {
          createUser.mutate(submitData, {
              onSuccess: () => {
                  handleOpenChange(false);
                  form.reset();
              }
          });
      }
  };
    
    const handleDialogChange = (isOpen: boolean) => {
        handleOpenChange(isOpen);
        if (!isOpen) {
            form.reset();
            setAuthType('Local');
        }
    };

    return (
        <Dialog open={isOpen} onOpenChange={handleDialogChange}>
            {children && (
                <DialogTrigger asChild>
                    {children}
                </DialogTrigger>
            )}
            <DialogContent className="sm:max-w-[900px] max-h-[90vh] p-0 overflow-y-auto">
                <DialogHeader className="bg-card border-b p-6">
                    <DialogTitle>
                        {user ? t('Edit User') : t('Create User')}
                    </DialogTitle>
                    <DialogDescription>
                        {user ? t('Update user information below.') : t('Please enter all fields below to create a new user.')}
                    </DialogDescription>
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                    <div className="p-6 flex flex-col gap-6">
                        {/* User Details Card */}
                        <Card>
                            <CardContent>
                                <h3 className="text-lg font-semibold mb-4">{t("User Details")}</h3>
                                <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Full Name")}</FormLabel>
                                        <FormDescription>
                                            {t("Name displayed in interface")}
                                        </FormDescription>
                                    </div>
                                    <div className="flex gap-2 w-md">
                                        <FormField
                                            control={form.control}
                                            name="first_name"
                                            render={({ field }) => (
                                                <FormItem className="w-full">
                                                    <FormControl>
                                                        <Input placeholder={t("First Name")} {...field} />
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                        <FormField
                                            control={form.control}
                                            name="last_name"
                                            render={({ field }) => (
                                                <FormItem className="w-full">
                                                    <FormControl>
                                                        <Input placeholder={t("Last Name")} {...field} />
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                    </div>
                                </FormItem>

                                <PageSeparator />

                                <FormField 
                                    control={form.control}
                                    name="company"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Company')}</FormLabel>
                                                <FormDescription>{t('Name displayed in Nagios Applications')}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <Input {...field} placeholder={t('Nagios Enterprises')} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <PageSeparator />

                                <FormField 
                                    control={form.control}
                                    name="email"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Email')}</FormLabel>
                                                <FormDescription>{t('Your email address for account access')}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <Input {...field} placeholder="example@domain.com" />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                                <PageSeparator />
                                <FormField 
                                    control={form.control}
                                    name="phone"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Phone Number')}</FormLabel>
                                                <FormDescription>{t('Contact phone number')}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <Input {...field} placeholder="+1 123-456-7890" />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </CardContent>
                        </Card>

                        {/* Account Information Card */}
                        <Card>
                            <CardContent className="pt-6">
                                <h3 className="text-lg font-semibold mb-4">{t("Account Information")}</h3>
                                
                                <FormField 
                                    control={form.control}
                                    name="username"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Username')}</FormLabel>
                                                <FormDescription>{t("Your unique identifier in Network Analyzer")}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <Input {...field} placeholder='nagiosadmin' disabled={!!user} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <PageSeparator />

                                {(authType === "Local" && !user) && (
                                    <>
                                        <FormField 
                                            control={form.control}
                                            name="password"
                                            render={({ field }) => (
                                                <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                    <div className="space-y-1">
                                                        <FormLabel>{t('Password')}</FormLabel>
                                                        <FormDescription>{t('Your secure password for authentication')}</FormDescription>
                                                    </div>
                                                    <FormControl className="w-md">
                                                        <PasswordInput {...field} />
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />

                                        <PageSeparator />

                                        <FormField 
                                            control={form.control}
                                            name="confirm_password"
                                            render={({ field }) => (
                                                <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                    <div className="space-y-1">
                                                        <FormLabel>{t('Confirm Password')}</FormLabel>
                                                        <FormDescription>{t('Re-enter your password')}</FormDescription>
                                                    </div>
                                                    <FormControl className="w-md">
                                                        <PasswordInput {...field} />
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                        <PageSeparator />
                                    </>
                                )}

                                <FormField
                                    control={form.control}
                                    name="lang"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t("Language")}</FormLabel>
                                                <FormDescription>
                                                    {t("Set the display language")}
                                                </FormDescription>
                                            </div>
                                            <Popover open={languageOpen} onOpenChange={setLanguageOpen}>
                                                <PopoverTrigger asChild>
                                                    <FormControl>
                                                        <Button
                                                            variant="outline"
                                                            role="combobox"
                                                            className={cn(
                                                                "w-50 justify-between",
                                                                !field.value && "text-muted-foreground"
                                                            )}
                                                        >
                                                            {field.value
                                                                ? languages.find(
                                                                    (lang) => lang.value === field.value
                                                                )?.label
                                                                : "Select language"}
                                                            <ChevronsUpDown className="opacity-50" />
                                                        </Button>
                                                    </FormControl>
                                                </PopoverTrigger>
                                                <PopoverContent className="w-50 p-0">
                                                    <Command>
                                                        <CommandInput placeholder="Search language..." />
                                                        <CommandList>
                                                            <CommandEmpty>
                                                                {t("No language found")}
                                                            </CommandEmpty>
                                                            <CommandGroup>
                                                                {languages.map((language) => (
                                                                    <CommandItem
                                                                        value={language.label}
                                                                        key={language.value}
                                                                        onSelect={() => {
                                                                            setLanguageOpen(false);
                                                                            form.setValue("lang", language.value);
                                                                        }}
                                                                    >
                                                                        {language.label}
                                                                        <Check
                                                                            className={cn(
                                                                                "ml-auto",
                                                                                language.value === field.value
                                                                                    ? "opacity-100"
                                                                                    : "opacity-0"
                                                                            )}
                                                                        />
                                                                    </CommandItem>
                                                                ))}
                                                            </CommandGroup>
                                                        </CommandList>
                                                    </Command>
                                                </PopoverContent>
                                            </Popover>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <PageSeparator />

                                <FormField
                                    control={form.control}
                                    name="theme"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('User Theme')}</FormLabel>
                                                <FormDescription>{t('Set the display theme')}</FormDescription>
                                            </div>
                                            <FormControl>
                                                <div className="w-50">
                                                    <ThemesDropdown field={field} />
                                                </div>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="pt-6">
                                <h3 className="text-lg font-semibold mb-2">{t("Authentication Settings")}</h3>
                                <p className="text-sm text-muted-foreground mb-4">
                                    {t('User accounts can be authenticated in many different ways either from your local database or external programs such as Active Directory or LDAP. You can set up external authentication servers in the LDAP/AD Integration settings.')}
                                </p>

                                <PageSeparator />
                                
                                <FormField
                                    control={form.control}
                                    name="type"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t("Authentication Type")}</FormLabel>
                                                <FormDescription>
                                                    {t("Choose how this user will authenticate.")}
                                                </FormDescription>
                                            </div>
                                            <FormControl>
                                                <Select
                                                    value={field.value}
                                                    onValueChange={(selectedVal) => {
                                                      setAuthType(selectedVal as 'Local' | 'LDAP' | 'AD');
                                                      field.onChange(selectedVal);
                                                      
                                                      // Set auth server immediately when auth type changes
                                                      if (selectedVal == "AD" && adServers.length > 0) {
                                                          form.setValue("auth_server_id", adServers[0].id, { shouldDirty: true });
                                                      } else if (selectedVal == "LDAP" && ldapServers.length > 0) {
                                                          form.setValue("auth_server_id", ldapServers[0].id, { shouldDirty: true });
                                                      } else {
                                                          form.setValue("auth_server_id", 0, { shouldDirty: true });
                                                      }
                                                  }}
                                                    disabled={!!user}
                                                >
                                                    <SelectTrigger className="w-50">
                                                        <SelectValue placeholder={t("Select authentication type")} />
                                                    </SelectTrigger>
                                                    <SelectContent>
                                                        <SelectGroup>
                                                            <SelectItem value="Local">{t("Local")}</SelectItem>
                                                            <SelectItem value="AD" disabled={adServers.length === 0}>{t("Active Directory")}</SelectItem>
                                                            <SelectItem value="LDAP" disabled={ldapServers.length === 0}>{t("LDAP")}</SelectItem>
                                                        </SelectGroup>
                                                    </SelectContent>
                                                </Select>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                                <PageSeparator />

                                {/* Auth Server Selection */}
                                {(authType === "LDAP" || authType === "AD") && (
                                    <>
                                        <FormField
                                            control={form.control}
                                            name="auth_server_id"
                                            render={({ field }) => (
                                                <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                    <FormLabel>
                                                        {authType === "LDAP" ? t('LDAP Server') : t('AD Server')}
                                                    </FormLabel>
                                                    <FormControl>
                                                        <Select
                                                            value={field.value?.toString()}
                                                            onValueChange={(selectedVal) => {
                                                                field.onChange(Number(selectedVal));
                                                            }}
                                                        >
                                                            <SelectTrigger className="w-md">
                                                                <SelectValue placeholder="Select a server" />
                                                            </SelectTrigger>
                                                            <SelectContent>
                                                                <SelectGroup className="w-full max-h-80">
                                                                    {(authType === "LDAP" ? ldapServers : adServers).map((server: AuthServer) => (
                                                                        <SelectItem key={server.id} value={server.id.toString()}>
                                                                            {server.name} ({authType === "LDAP" ? server.host : server.controllers})
                                                                        </SelectItem>
                                                                    ))}
                                                                </SelectGroup>
                                                            </SelectContent>
                                                        </Select>
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />

                                        <PageSeparator />

                                        <FormField
                                            control={form.control}
                                            name="auth_server_data"
                                            render={({ field }) => (
                                                <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                    <FormLabel>
                                                        {authType === "LDAP"
                                                            ? t("User's Full DN")
                                                            : t("AD Username")}
                                                    </FormLabel>
                                                    <FormControl className="w-md">
                                                        <Input
                                                            {...field}
                                                            placeholder={
                                                                authType === "LDAP"
                                                                    ? "cn=John Smith,dn=nagios,dc=com"
                                                                    : t("AD Username")
                                                            }
                                                        />
                                                    </FormControl>
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                        <PageSeparator />
                                    </>
                                )}

                                {/* User Level Access */}
                                <FormField 
                                    control={form.control}
                                    name="is_admin"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('User Level Access')}</FormLabel>
                                                <FormDescription>{t("Set the user level of access inside the UI. Admin's have full access while User's can view components but only edit their own profile information.")}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <RadioGroup
                                                    className="gap-4 grid"
                                                    onValueChange={field.onChange}  
                                                    value={field.value}
                                                >
                                                    <div className="flex items-center space-x-2">
                                                        <RadioGroupItem value="1" id="user-option-one" />
                                                        <Label htmlFor="user-option-one">{t('Admin')}</Label>
                                                    </div>
                                                    <div className="flex items-center space-x-2">
                                                        <RadioGroupItem value="0" id="user-option-two" />
                                                        <Label htmlFor="user-option-two">{t('User')}</Label>
                                                    </div>
                                                </RadioGroup>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <PageSeparator />

                                <FormField 
                                    control={form.control}
                                    name="apiaccess"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('API Access')}</FormLabel>
                                                <FormDescription>{t('If you want to allow this user to use the external API via an access key')}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <RadioGroup
                                                    className="gap-4 grid"
                                                    onValueChange={field.onChange}  
                                                    value={field.value}
                                                >
                                                    <div className="flex items-center space-x-2">
                                                        <RadioGroupItem value="1" id="api-option-one" />
                                                        <Label htmlFor="api-option-one">{t('Yes')}</Label>
                                                    </div>
                                                    <div className="flex items-center space-x-2">
                                                        <RadioGroupItem value="0" id="api-option-two" />
                                                        <Label htmlFor="api-option-two">{t('No')}</Label>
                                                    </div>
                                                </RadioGroup>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </CardContent>
                        </Card>
                        </div>

                        <DialogFooter className="bg-card p-6">
                            <Button 
                                type="button" 
                                variant="secondary" 
                                onClick={() => handleOpenChange(false)}
                            >
                                {t("Cancel")}
                            </Button>
                            <Button type="submit">
                                {user ? t('Update User') : t('Create User')}
                            </Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
}