import { useTranslation } from "react-i18next";
import { toast } from "sonner";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import axios, { AxiosError } from "axios";
import { BaseUser } from "./types";
import { User } from "./types";
import { useNavigate } from "react-router-dom";

export function useGetUsers() {
    return useQuery<BaseUser[], AxiosError>({
        queryKey: ["users"],
        queryFn: async () => {
            const response = await axios.get("/api/v1/user");
            return response.data;
        },
        retry: false,
    });
}

export function useCreateUser() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const navigate = useNavigate();

    return useMutation({
        mutationFn: (data: User) => {
            return axios.post("/api/v1/user", data);
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["users"] });
            navigate("/admin/users");
            toast(t("User Created."));
        },
        onError: () => {
            toast(t("Failed to create user."), {
                description: t("Please try again."),
            });
        },
    });
}

export function useEditUser() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: ({ data }: { data: User }) => {
            return axios.put("/api/v1/user", data);
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["users"] });
            toast(t("User edited."));
        },
        onError: () => {
            toast(t("Failed to edit user."), {
                description: t("Please try again."),
            });
        },
    });
}

export function useDeleteUser() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (id: number) => axios.delete(`/api/v1/user/${id}`),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["users"] });
            toast(t("User deleted."));
        },
        onError: () => {
            toast(t("Failed to delete user."), {
                description: t("Please try again"),
            });
        },
    });
}