import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import {  WiresharkScan, } from '@/components/wireshark/capturehistory/types';
const API_BASE_History = '/api/v1/wireshark/scan';

export function useGetWiresharkScans() {
    return useQuery<WiresharkScan[], AxiosError>({
        queryKey: ['wiresharkScansHistory'],
        queryFn: async () => {
        const response = await axios.get('/api/v1/wireshark/scan');
        return response.data.scans;
        },
    });
}

export function useDeleteCaptures() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (scanIds: number[]) =>
      axios.delete(`${API_BASE_History}/`, {
        data: { ids: scanIds },
      }),

    onMutate: async (scanIds) => {
      const key = ['wiresharkScansHistory'];
      await queryClient.cancelQueries({ queryKey: key });

      const previousScans = queryClient.getQueryData<WiresharkScan[]>(key);

      queryClient.setQueryData(key, (old: WiresharkScan[] = []) =>
        old.filter(scan => !scanIds.includes(scan.id))
      );

      return { previousScans };
    },

    onSuccess: (_data, scanIds) => {
      const message = scanIds.length > 1
        ? t('Scans deleted successfully.')
        : t('Scan deleted successfully.');
      toast.success(message);
    },

    onError: (_error, scanIds, context) => {
      queryClient.setQueryData(['wiresharkScansHistory'], context?.previousScans);
      const errorMessage = scanIds.length > 1
        ? t('Failed to delete scans.')
        : t('Failed to delete scan.');
      toast.error(errorMessage, {
        description: t('Please try again.'),
      });
    },

    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['wiresharkScansHistory'] });
      queryClient.refetchQueries({ queryKey: ['wiresharkScansHistory'] });
    },
  });
}

export function useDownloadCapture() {
  return useMutation({
    mutationFn: async (scanId: number) => {
      const response = await axios.get(`${API_BASE_History}/download/${scanId}`, {
        responseType: 'blob',
      });
      return { blob: response.data, scanId };
    },
  });
}

export function useScanWithSuricata() {
  return useMutation({
    mutationFn: async (scanId: number) => {
      const response = await axios.post(
        `${API_BASE_History}/scanWithSuricata`,
        { id: scanId }
      );
      toast.info("Scanning pcap with Suricata")
      return response.data;
    },
  });
}

export function useImportCapture(options?: { onSuccess?: () => void }) {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (formData: FormData) => {
      const response = await axios.post(`${API_BASE_History}/import`, formData);
      return response.data;
    },
    onSuccess: () => {
      toast.success(t('PCAP file uploaded successfully.'));
      queryClient.invalidateQueries({ queryKey: ['wiresharkScansHistory'] }); // Refresh history
      options?.onSuccess?.(); // Close dialog, reset file, etc.
    },
    onError: (error: AxiosError) => {
      toast.error(t('Failed to upload PCAP file'), {
        description: error.message,
      });
    },
  });
}
