import { useQuery} from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import {Packet } from '@/components/wireshark/captures/types';


const API_BASE = '/api/v1/wireshark';
interface HistoryPacketResponse {
  file_name: string;
  packets: Packet[];
  total: number; 
}

interface UseHistoryParams {
  id: string
  pageIndex: number
  pageSize: number
  sortBy: string
  sortOrder: 'asc' | 'desc'
  filter?: string
}
export function useGetHistoryPackets({
  id,
  pageIndex,
  pageSize,
  sortBy,
  sortOrder,
  filter,
}: UseHistoryParams) {
  return useQuery<HistoryPacketResponse, AxiosError>({
    queryKey: [
      'wiresharkCapture',
      id,
      pageIndex,
      pageSize,
      sortBy,
      sortOrder,
      filter,
    ],
    queryFn: async () => {
      const response = await axios.get<HistoryPacketResponse>(
        `${API_BASE}/gethistorypackets/${id}`,
        {
          params: {
            page:      pageIndex + 1,
            per_page:  pageSize,
            sortBy,
            sortOrder,
            ...(filter ? { filter } : {}),
          },
          paramsSerializer: params =>
            Object.entries(params)
              .map(([key, value]) => `${key}=${encodeURIComponent(value as string)}`)
              .join('&'),
        }
      )
      return response.data
    },
    enabled: !!id,
  })
}
