import { useTranslation } from 'react-i18next';
import { useGetCaptureInfo, useGetProtocolUsage, useGetTopTalkers, useGetPacketSizeDistribution } from '@/components/wireshark/capturesummary/queries';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Hash, Binary, Clock, Server } from "lucide-react";
import { ProtocolPieChart } from "@/components/wireshark/capturesummary/ProtocolPieChart";
import { PacketSizeBarChart } from "@/components/wireshark/capturesummary/PacketSizeBarChart";
import { Skeleton } from "@/components/ui/skeleton"
import InfoCard from '@/components/ui/InfoCard';

interface CaptureSummaryProps {
  captureId: string;
}

function formatBytes(bytes: number): string {
  if (bytes === 0) return "0 B";
  const k = 1024;
  const sizes = ["B", "KB", "MB", "GB", "TB"];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
}
function formatDuration(duration: string) {
  const [hours, minutes, seconds] = duration.split(':').map(Number);
  let result = '';
  if (hours > 0) result += `${hours}h `;
  if (minutes > 0) result += `${minutes}m `;
  if (seconds > 0 || result === '') result += `${seconds}s`;
  return result.trim();
}

export default function CaptureSummary({ captureId }: CaptureSummaryProps) {
  const { t } = useTranslation();

  const {
    data: protocolUsage,
    isLoading: isLoadingProtocols,
    error: protocolError,
  } = useGetProtocolUsage(captureId);

  const {
    data: captureInfo,
    isLoading: isLoadingInfo,
    error: infoError,
  } = useGetCaptureInfo(captureId);

  const {
    data: topTalkers,
    isLoading: isLoadingTalkers,
    error: talkerError,
  } = useGetTopTalkers(captureId);

  const {
    data: packetSizeDistribution,
    isLoading: isLoadingPacketSize,
    error: packetSizeError,
  } = useGetPacketSizeDistribution(captureId);

  if (isLoadingProtocols || isLoadingInfo || isLoadingTalkers || isLoadingPacketSize) {
    return (
      <Card>
        <CardContent>
          <div className="w-full flex flex-col gap-4">
            <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-4 gap-4">
              {Array.from({ length: 4 }).map((_, idx) => (
                <Skeleton
                  key={idx}
                  className="h-[164px] w-full rounded-xl"
                />
              ))}
            </div>
            <div className="w-full flex flex-col lg:flex-row gap-4">
              <div className="flex flex-col gap-4 w-full lg:w-1/3">
                <Skeleton className="h-[438px] w-full rounded-xl" />
                <Skeleton className="h-[438px] w-full rounded-xl" />
              </div>
              <div className="flex flex-col gap-4 w-full lg:w-2/3">
                <Skeleton className="h-[438px] w-full rounded-xl" />
                <Skeleton className="h-[438px] w-full rounded-xl" />
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    );
  }

  if (protocolError || infoError || talkerError || packetSizeError) {
    return (
      <Card>
        <CardContent>{t('Failed to load scan.')}</CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardContent>
        <div className="w-full flex flex-col gap-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-4 gap-4">
            <InfoCard
              header={t('Interface')}
              data={captureInfo ? captureInfo.interface : ' '}
              icon={<Server />}
              tooltip={t('Network interface used for the capture.')}
            />
            <InfoCard
              header={t('Duration')}
              data={captureInfo ? formatDuration(captureInfo.duration) : '0 B'}
              icon={<Clock />}
              tooltip={t('Total duration of the capture.')}
            />
            <InfoCard
              header={t('Total Packets')}
              data={captureInfo?.frames.toString() || '0'}
              icon={<Hash />}
              tooltip={t('Total number of packets.')}
            />
            <InfoCard
              header={t('Total Bytes')}
              data={captureInfo ? formatBytes(captureInfo.bytes) : '0 B'}
              icon={<Binary />}
              tooltip={t('Total number of bytes.')}
            />
          </div>
          <div className="w-full flex flex-col lg:flex-row gap-4">
            <div className="w-full lg:w-1/3 flex flex-col gap-4">
              <Card className="w-full h-full">
                <CardHeader>
                  <CardTitle>{t('Top 10 Talkers (Source)')}</CardTitle>
                  <CardDescription>{t('Top 10 talkers by bytes')}</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex justify-between text-sm font-semibold border-b pb-1 mb-2">
                    <span className="font-medium">{t('Source IP')}</span>
                    <span className="font-medium">{t('Bytes')}</span>
                  </div>
                  <ul className="space-y-1 text-sm">
                    {topTalkers?.top_transmitters.map((talker) => (
                      <li
                        key={talker.ip}
                        className="flex justify-between border-b pb-1 last:border-none"
                      >
                        <span className="">{talker.ip}</span>
                        <span className="text-muted-foreground">
                          {talker.total_bytes}
                        </span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
              <Card className="w-full h-full">
                <CardHeader>
                  <CardTitle>{t('Top 10 Talkers (Destination)')}</CardTitle>
                  <CardDescription>{t('Top 10 talkers by bytes')}</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex justify-between text-sm font-semibold border-b pb-1 mb-2">
                    <span className="font-medium">{t('Destination IP')}</span>
                    <span className="font-medium">{t('Bytes')}</span>
                  </div>
                  <ul className="space-y-1 text-sm">
                    {topTalkers?.top_receivers.map((talker) => (
                      <li
                        key={talker.ip}
                        className="flex justify-between border-b pb-1 last:border-none"
                      >
                        <span className="">{talker.ip}</span>
                        <span className="text-muted-foreground">
                          {talker.total_bytes}
                        </span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            </div>
            <div className="w-full lg:w-2/3 flex flex-col gap-4">
              <div className="w-full">
                <ProtocolPieChart protocolUsage={protocolUsage || []} />
              </div>
              <div className="w-full">
                <PacketSizeBarChart packetSizeDistribution={packetSizeDistribution || []} />
              </div>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}