import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { useEcho } from '@laravel/echo-react';
import { toast } from 'sonner';
import { useQueryClient } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { useNavigate } from "react-router-dom"
import { Card } from "@/components/ui/card";
import { Server, Hash, FileText, Clock, LoaderCircle, Eye, CheckCircle, XCircle, Download } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import type { VariantProps } from "class-variance-authority";
import { badgeVariants } from "@/components/ui/badge";
import { useTranslation } from "react-i18next"
import {Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tooltip, TooltipTrigger, TooltipContent } from "@/components/ui/tooltip";
import { useGetNetworkInterfaces, useStartScan, useStopScan, useLatestScan } from '@/components/wireshark/livecapture/queries';
import { useDownloadCapture } from "@/components/wireshark/capturehistory/queries";
import { NetworkInterface } from '@/components/wireshark/livecapture/types';
import  InfoCard  from '@/components/ui/InfoCard';
import { Skeleton } from '@/components/ui/skeleton';
import { Duration } from '@/components/ui/Duration';
import { useAuth } from '@/contexts/AuthContext';

const FormSchema = z.object({
  networkInterface: z.string({
    required_error: 'Please select a network interface to scan.',
  }),
  duration: z.string({
    required_error: 'Please select a capture duration.',
  }),
});

type FormValues = z.infer<typeof FormSchema>;
type BadgeVariant = VariantProps<typeof badgeVariants>["variant"]

function formatDateTime(value?: string | null) {
  if (!value) return " ";
  const [datePart, timePart] = value.split("T");
  const [year, month, day] = datePart.split("-");
  const [hour, minute, second] = timePart.split(/[:\-]/);
  return `${year}-${month}-${day} ${hour}:${minute}:${second}`;
}

export function WiresharkScan() {
  const queryClient = useQueryClient();
  const navigate = useNavigate();
  const { user } = useAuth();

  const { t } = useTranslation();
  const form = useForm<FormValues>({
    resolver: zodResolver(FormSchema),
    defaultValues: {
      networkInterface:'',
      duration: '',
    },
  });

  const { data: interfacesData } = useGetNetworkInterfaces();
  const interfaces: NetworkInterface[] = interfacesData || [];

  const startScanMutation = useStartScan();
  const stopScanMutation = useStopScan();
  const { mutate: downloadCapture } = useDownloadCapture();
  const { data: activeScan , isLoading} = useLatestScan();

  // Live updates from Echo broadcast
  useEcho(`NNA.Admins`, 'PacketCaptured', () => {
    queryClient.invalidateQueries({ queryKey: ['activescan'] });
  });

  const scan = activeScan ?? {
    id: null,
    interface: '',
    status: '',
    file_name: '',
    started_at: '',
    finished_at: null,
    pid: null,
  };

  const fileNameOnly = scan.file_name ? scan.file_name.split(/[/\\]/).pop() : " ";

  const handleStartScan = (values: FormValues) => {
    if (!values.networkInterface) {
      toast.warning(t('Please select a network interface before starting the capture.'));
      return;
    }
    if (!values.duration) {
      toast.warning(t('Please select a duration before starting the capture.'));
      return;
    }
    if (scan.status === 'in_progress') {
      toast.warning(t('Scan already in progress. Stop or restart before starting a new one.'));
      return;
    };
    startScanMutation.mutate({ networkInterface: values.networkInterface, duration: values.duration }, {
      onSuccess: () => {
        queryClient.invalidateQueries({ queryKey: ['activescan'] });
      },
    });
  };

  const handleStopScan = () => {
    if (!scan.id) {
      toast.error(t('No active scan to stop.'));
      return;
    }

    stopScanMutation.mutate(scan.id);
  };

  const handleView = () => {
      navigate(`/wireshark/${scan.id}`);
  };

  const handleDownload = (scanId:number ) => {
    if (!scanId) {
      toast.error(t("No scan ID available for download."));
      return;
    }
      downloadCapture(scanId, {
        onSuccess: ({ blob, scanId }) => {
          const url = window.URL.createObjectURL(new Blob([blob]));
          const link = document.createElement("a");
          link.href = url;
          link.setAttribute("download", `capture_${scanId}.pcap`);
          document.body.appendChild(link);
          link.click();
          document.body.removeChild(link);
          window.URL.revokeObjectURL(url);
          toast.success(t("Capture file downloaded successfully."));
        },
        onError: () => {
          toast.error(t("Failed to download capture file."));
        },
      });
  };

  let variant: BadgeVariant = "default"
  let status= "Running"

  if (scan.status === "completed") {
    variant = "green";
    status="Completed";
  } else if (scan.status === "in_progress") {
    variant = "blue";
    status="Running"
  } else if (scan.status === "pending"){
    variant = "blue";
    status="Pending"
  } else if (scan.status === "failed") {
    variant = "red";
    status="Failed"
  } else {
    status="No Capture Running"
  }

  if (!user?.is_admin) return <div>{t("Only admin users can start a live scan. If you need to do so to view data please contact an admin user to start the scan.")}</div>

  return (
    <div className="hidden h-full flex-1 flex-col space-y-8 py-2 md:flex relative">
      <Form {...form}>
        <form onSubmit={form.handleSubmit(handleStartScan)} className="w-full max-w-xl space-y-4">
          <div className="flex items-end gap-3 flex-wrap">
            <FormField
              control={form.control}
              name="networkInterface"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Network Interface')}</FormLabel>
                  <Select onValueChange={field.onChange} value={field.value}>
                    <FormControl>
                      <SelectTrigger className="w-50">
                        <SelectValue placeholder={t("Select a Network Interface to scan")} />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {interfaces.map((iface) => (
                        <SelectItem key={iface.id} value={iface.name}>
                          {iface.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="duration"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Duration')}</FormLabel>
                  <Select onValueChange={field.onChange} value={field.value}>
                    <FormControl>
                      <SelectTrigger className="w-50">
                        <SelectValue placeholder={t("Select a Duration")} />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      <SelectItem value="30">{t("30 Seconds")}</SelectItem>
                      <SelectItem value="60">{t("1 Minute")}</SelectItem>
                      <SelectItem value="180">{t("3 Minutes")}</SelectItem>
                      <SelectItem value="300">{t("5 Minutes")}</SelectItem>
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
            <div className="flex gap-4">
              <Button
                type="button"
                onClick={() => {
                  if (scan.status === 'in_progress') {
                    handleStopScan();
                  } else {
                    form.handleSubmit(handleStartScan)();
                  }
                }}
                disabled={scan.status === 'pending'||(scan.status === 'in_progress' && !scan.pid)}
              >
                {scan.status === 'pending'
                  ? t('Starting Capture...')
                  : scan.status === 'in_progress'
                    ? t('Stop Capture')
                    : t('Start Capture')}
              </Button>
            </div>
          </div>
        </form>
      </Form>
      { isLoading ? (
        <Card className="w-full h-full min-h-[280px] p-4 md:p-5 shadow-lg border border-border bg-background rounded-xl relative space-y-4">
          <div className="flex items-center gap-3 mb-2">
            <Skeleton className="h-7 w-40 rounded-md" />
            <div className="ml-auto flex items-center gap-1">
              <Skeleton className="h-9 w-9 rounded-md" />
              <Skeleton className="h-9 w-9 rounded-md" />
            </div>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-x-6 gap-y-8 text-sm">
            {Array.from({ length: 6 }).map((_, i) => (
              <Skeleton
                key={i}
                className="h-[174px] w-full rounded-xl" 
              />
            ))}
          </div>
        </Card>
      ) : (
        scan.id && (
            <Card className="w-full h-full min-h-[280px] p-4 md:p-5 shadow-lg border border-border bg-background rounded-xl relative space-y-4">
              <div className="flex items-center gap-3 mb-2">
                <h1 className="text-2xl font-bold tracking-tight text-foreground">
                  {t('Latest Capture :')}
                </h1>
                <div className="text-2xl font-semibold ">
                  <Duration started_at={scan.started_at as string | undefined} finished_at={scan.finished_at as string | undefined} />
                </div>
                <div className="ml-auto flex items-center gap-1">
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Button
                        variant="secondary"
                        aria-label={t('View Capture')}
                        onClick={handleView}
                        disabled={!scan.id || scan.status === 'in_progress' || scan.status === 'pending'}
                      >
                        <Eye className="w-5 h-5" />
                      </Button>
                    </TooltipTrigger>
                    <TooltipContent>{t('View Capture')}</TooltipContent>
                  </Tooltip>
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Button
                        variant="secondary"
                        aria-label={t('Download Capture')}
                        onClick={() => handleDownload(scan.id ?? 0)}
                        disabled={!scan.id || scan.status === 'in_progress' || scan.status === 'pending'}
                      >
                        <Download className="w-5 h-5" />
                      </Button>
                    </TooltipTrigger>
                    <TooltipContent>{t('Download Capture')}</TooltipContent>
                  </Tooltip>
                </div>
              </div>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-x-6 gap-y-8 text-sm">
                <InfoCard
                  header={t('Interface')}
                  data={scan.interface || " "}
                  icon={<Server />}
                  tooltip={t('Network interface used for the capture.')}
                />
                <InfoCard
                  header={t('Scan ID')}
                  data={scan.id ?? " "}
                  icon={<Hash />}
                  tooltip={t('Unique identifier for the scan.')}
                />
                <InfoCard
                  header={t('Status')}
                  data={
                    <Badge variant={variant} className="mt-1 text-lg px-3 py-1 rounded-md">
                      {status || " "}
                    </Badge>
                  }
                  icon={scan.status === 'in_progress' || scan.status === 'pending' ? (
                      <div className="flex items-center gap-2">
                        <LoaderCircle className="animate-spin w-5 h-5" />
                      </div>
                    ) : scan.status === 'completed' ? (
                      <div className="flex items-center gap-2">
                        <CheckCircle className="w-5 h-5" />
                      </div>
                    ) : scan.status === 'failed' ? (
                      <div className="flex items-center gap-2">
                        <XCircle className="w-5 h-5" />
                      </div>
                    ) : (
                      <span>{t('Unknown')}</span>
                    )}
                  tooltip={t('Current status of the scan.')}
                />
                <InfoCard
                  header={t('Started')}
                  data={formatDateTime(scan.started_at)}
                  icon={<Clock />}
                  tooltip={t('Timestamp when the scan started.')}
                />
                <InfoCard
                  header={t('Finished')}
                  data={formatDateTime(scan.finished_at)}
                  icon={<Clock />}
                  tooltip={t('Timestamp when the scan finished.')}
                />
                <InfoCard
                  header={t('Filename')}
                  data={fileNameOnly || " "}
                  icon={<FileText />}
                  tooltip={t('Name of the file where the capture is saved.')}
                />
              </div>
            </Card>
          )
        )}
    </div>
  );
}
