import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import { NetworkInterface, WiresharkScan} from '@/components/wireshark/livecapture/types';

export function useGetNetworkInterfaces() {
  return useQuery<NetworkInterface[], AxiosError>({
    queryKey: ['NetworkInterfaces'],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/wireshark/getinterface`);
      return response.data.interfaces;
    },
  });
}
export function useLatestScan() {
    return useQuery<WiresharkScan | null, AxiosError>({
        queryKey: ['activescan'],
        queryFn: async () => {
        const response = await axios.get(`/api/v1/wireshark/latestscan`);
        return response.data.scan ?? null;
        },
        staleTime: 10000,
    });
}

export function useStartScan() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: ({ networkInterface, duration }: { networkInterface: string; duration: string }) =>
      axios.post(`/api/v1/wireshark/startscan`, { interface: networkInterface, duration: duration, }),
    onSuccess: () => {
      toast.success(t('Capture started.'));
      queryClient.invalidateQueries({ queryKey: ['wiresharkscan'] }); 
    },
    onError: () => {
      toast.error(t('Failed to start capture.'));
    },
  });
}

export function useStopScan() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (processId: number) =>
      axios.post(`/api/v1/wireshark/stopscan`, { id: processId }),
    onSuccess: () => {
      toast.success(t('Capture stopped.'));
      queryClient.invalidateQueries({ queryKey: ['wiresharkscan'] });
    },
    onError: (error: AxiosError) => {
      toast.error(t('Failed to stop scan.'), {
        description: error.message,
      });
    },
  });
}