import { useTranslation } from 'react-i18next';
import InfoCard from '@/components/ui/InfoCard';
import { Card, CardContent } from '@/components/ui/card';
import { Hash, Scan, Clock, Server } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton"
import { useWiresharkOverview } from './queries';
import { WiresharkScansOverTimeBarChart } from './ScansOverTimeChart';
import { OverviewPieChart } from './OverviewPIeChart';
import { useNavigate } from "react-router-dom"

function formatDuration(duration?: number) {
	if (!duration || duration <= 0) return '0s';

	const hours = Math.floor(duration / 3600);
	const minutes = Math.floor((duration % 3600) / 60);
	const seconds = +Math.floor((duration % 60));

	if (hours > 0) return `${hours}h ${minutes}m ${seconds}s`;
	if (minutes > 0) return `${minutes}m ${seconds}s`;
	return `${seconds}s`;
}

export default function Overview() {
	const { t } = useTranslation();
	const { data: overviewData, isLoading: isLoadingOverview } = useWiresharkOverview();
	const navigate = useNavigate();

	if (isLoadingOverview) {
    return (
      <Card>
        <CardContent>
          <div className="w-full flex flex-col gap-4">
            <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-4 gap-4">
              {Array.from({ length: 4 }).map((_, idx) => (
                <Skeleton
                  key={idx}
                  className="h-[164px] w-full rounded-xl"
                />
              ))}
            </div>
            <Skeleton className="h-[338px] w-full rounded-xl" />
            <div className="flex flex-col lg:flex-row gap-4">
              <Skeleton className="h-[364px] w-full rounded-xl" />
              <Skeleton className="h-[364px] w-full rounded-xl" />
            </div>
          </div>
        </CardContent>
      </Card>
    );
	}

	return (
    <Card>
      <CardContent>
        <div className="w-full flex flex-col gap-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-4 gap-4">
            <InfoCard
              header={t('Total Captures')}
              data={overviewData?.totalscan.toString() || '0'}
              icon={<Hash />}
              tooltip={t('Total number of Captures.')}
            />
            <InfoCard
              header={t('Average Duration')}
              data={formatDuration(overviewData?.average_duration) || '0'}
              icon={<Clock />}
              tooltip={t('Average duration of Captures.')}
            />
            <InfoCard
              header={t('Most Used Interface')}
              data={overviewData?.most_used_interface || 'None'}
              icon={<Server />}
              tooltip={t('The network interface used most for captures.')}
            />
            <InfoCard
              className={`transition-transform 
                      ${overviewData?.most_recent_scan?.id ? "cursor-pointer rounded-xl hover:bg-accent hover:text-white hover:shadow-lg hover:scale-102" : "opacity-50 pointer-events-none"}`}
              onClick={() => navigate(`/wireshark/${overviewData?.most_recent_scan?.id}`)}
              header={t('Most Recent Capture ID')}
              data={overviewData?.most_recent_scan?.id || '0'}
              icon={<Scan />}
              tooltip={t('The most recent capture.')}
            />
          </div>
          <div className="grid grid-cols-1 xl:grid-cols-3 gap-4">
            <WiresharkScansOverTimeBarChart className='w-full' />
            <div className="grid grid-cols-1 lg:grid-cols-2 xl:col-span-2 gap-4">
              <OverviewPieChart
                title="Interface Usage"
                description="Captures per network interface"
                data={overviewData?.interface_usage ?? {} as Record<string, number>}
                className='w-full'
              />
              <OverviewPieChart
                title="Duration Distribution"
                description="Number of captures by duration"
                data={overviewData?.duration_distribution ?? {} as Record<string, number>}
                className='w-full'
              />
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
	);
}