import { GetRingBuffer, useStopRingBuffer } from '@/components/wireshark/ringbuffer/queries';
import { useTranslation } from 'react-i18next';
import { StartRingBufferDialog } from '@/components/wireshark/ringbuffer/StartRingBufferDialog';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Hash, Network, Folder, FileStack, Database, File, Square}  from 'lucide-react';
import type { VariantProps } from "class-variance-authority";
import { Badge, badgeVariants } from "@/components/ui/badge";
import { Tooltip, TooltipTrigger, TooltipContent } from "@/components/ui/tooltip";
import {Eye} from "lucide-react";
import { useNavigate } from "react-router-dom"
import { Skeleton } from '@/components/ui/skeleton';
import { useAuth } from '@/contexts/AuthContext';
import AddEditRingBufferDialog from '@/components/wireshark/ringbuffer/AddEditRingBufferDialog';
import InfoCard from '@/components/ui/InfoCard';

type BadgeVariant = VariantProps<typeof badgeVariants>["variant"]

export default function RingBuffer() {
  const { t } = useTranslation();
  const navigate = useNavigate();
  const { data, isLoading, error } = GetRingBuffer();
  const stopRingBuffer = useStopRingBuffer();
  const { user } = useAuth();

  const truncate = (str: string, max: number) => str.length > max ? str.slice(0, max) + "…" : str;

  let variant: BadgeVariant = "default";
  let status= t("none");
  if (data !== "none" && data?.status === "running") {
    variant = "green";
    status=t("Active");
  }  else if (data !== "none" && data?.status === "stopped") {
    variant = "red";
    status=t("Inactive");
  } else {
    status=t("No Capture Running");
  }

  const handleView = () => {
      navigate(`/wireshark/ringbuffer`);
  };

  if (error) return <p className="text-red-500">{t('Error:')} {error.message}</p>

  return (
    <div>
      {data === 'none' ? (
        <div className="space-y-4">
          {!!user?.is_admin && <AddEditRingBufferDialog/>}
          <Card className="mt-6 gap-2">
            <CardHeader>
              <CardTitle className="text-xl font-bold ml-3">
              {t("Ring Buffer is not setup  on this system.")}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex pb-2">
                <Card className="w-1/2 m-2">
                  <CardHeader>
                    <CardTitle className="text-xl font-bold">{t('Description')}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-bg mt-2">
                      {t("A ring buffer is a continuous packet capture method that stores data across multiple files.")}
                      <br />
                      {t("When the maximum number of files or file size is reached, the oldest capture files are automatically overwritten.")}
                      <br />
                      {t("This allows for long-term network monitoring without exhausting disk space.")}
                    </p>
                  </CardContent>
                </Card>
                <Card className="w-1/2 m-2">
                  <CardHeader>
                    <CardTitle className="text-xl font-bold">{t('Features')}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <ul className="list-disc pl-6 mt-2">
                      <li>{t('Prevents disk space from filling up with capture files')}</li>
                      <li>{t('Ensures the most recent network traffic is always available')}</li>
                      <li>{t('Ideal for long-running captures and continuous monitoring')}</li>
                      <li>{t('No manual cleanup needed: old files are rotated out automatically')}</li>
                    </ul>
                  </CardContent>
                </Card>
              </div>
            </CardContent>
          </Card>
        </div>
      ) : (
        <>
          <Card>
            <CardContent>
              <div className="flex items-center gap-4">
                {isLoading ? (
                  <Skeleton className="h-10 w-10 rounded-lg" />
                ) : (
                  <>
                    {!!user?.is_admin && ( 
                      data?.status === "running" ? (
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button
                            variant="destructive"
                            className="w-10"
                            onClick={() => data?.id && stopRingBuffer.mutate(data.id)}
                          >
                            <Square fill="white" />
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p>{t('Stop')}</p>
                        </TooltipContent>
                      </Tooltip>
                    ) : (
                      <StartRingBufferDialog id={data?.id || 0} />
                    )
                    )}
                  </>
                )}

                {isLoading ? (
                  <Skeleton className="h-9 rounded-full" />
                ) : (
                  <Badge
                    variant={variant}
                    className="text-base h-9 flex items-center px-4 rounded-full"
                  >
                    {status}
                  </Badge>
                )}

                <div className="flex items-center gap-2 ml-auto">
                  {isLoading ? (
                    <>
                      <Skeleton className="h-10 w-10 rounded-lg" />
                      <Skeleton className="h-10 w-10 rounded-lg" />
                    </>
                  ) : (
                    <>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button size="icon" onClick={handleView}>
                            <Eye className="w-10 h-10 " />
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p>{t('View the latest time period')}</p>
                        </TooltipContent>
                      </Tooltip>
                      {!!user?.is_admin && <AddEditRingBufferDialog ringbuffer={data} />}
                    </>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
          {isLoading ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-x-6 gap-y-8 mt-4 text-sm">
              {Array.from({ length: 6 }).map((_, i) => (
                <Skeleton
                  key={i}
                  className="h-[164px] w-full rounded-xl" 
                />
              ))}
            </div>
          ) : (
            <div className="mt-5 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
              {[
                { icon: <Hash />, header: t('ID'), data: data?.id, tooltip: t('Unique identifier for this ring buffer.') },
                { icon: <Network />, header: t('Interfaces'), data: data?.interfaces?.join(', '), tooltip: t('Network interfaces being captured.') },
                { icon: <Folder />, header: t('Directory'), data: truncate(data?.directory ?? "", 20), tooltip: t('Where capture files are stored.') },
                { icon: <FileStack />, header: t('Files'), data: data?.number_of_files, tooltip: t('Number of files in the ring buffer.') },
                { icon: <File />, header: t('File Size'), data: data?.file_size && `${data.file_size} KB`, tooltip: t('Max size of each file.') },
                { icon: <Database />, header: t('Ring Buffer Size'), data: data?.total_file_size && `${data.total_file_size} KB`, tooltip: t('Total storage allocated to the buffer.') },
              ].map((item, idx) => (
                <InfoCard
                  key={idx}
                  header={item.header}
                  data={
                  String(item.data ?? "—")
                  }
                  icon={item.icon}
                  tooltip={item.tooltip}
                />
              ))}
            </div>
          )}
        </>
      )}
    </div>
  )
}