import { useMutation, useQuery, useQueryClient} from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { toast } from 'sonner';
import { RingBuffer  } from '@/components/wireshark/ringbuffer/types';
import { useTranslation } from 'react-i18next';
import { Packet } from "@/components/wireshark/captures/types";

export function GetRingBuffer(){
    return useQuery<RingBuffer|'none',AxiosError>({
        queryKey: ['ringbuffer'],
        queryFn: async() => {
            const response = await axios.get('/api/v1/wireshark/ringbuffer/');
            return response.data.ringbuffer;
        },
    });
}

export function useCreateRingBuffer(){
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (payload: {
      interfaces: string[];
      number_of_files: number;
      file_size: number;
      directory: string;
      file_name: string;
    }) => axios.post('/api/v1/wireshark/ringbuffer/create', payload),
    onSuccess:() => {
      toast.success(t('Ring Buffer Created.'))
      queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.error || error.message);
    },
  });
}

export function useStopRingBuffer(){
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (ringbufferid: number) => 
      axios.post('/api/v1/wireshark/ringbuffer/stop',{id: ringbufferid}),
    onSuccess:() => {
      toast.success(t('Ring Buffer Stopped.'))
      queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.error || error.message);
    },
  });
}

export function useEditRingBuffer(){
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (payload: {
      id:number;
      interfaces: string[];
      number_of_files: number;
      file_size: number;
      directory: string;
      file_name: string;
    }) => axios.post('/api/v1/wireshark/ringbuffer/edit', payload),
    onSuccess:() => {
      toast.success(t('Ring Buffer Updated.'))
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.error || error.message);
    },
  });
}

export function useStartRingBuffer(){
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (ringbufferid: number) => 
      axios.post('/api/v1/wireshark/ringbuffer/start',{id: ringbufferid}),
    onSuccess:() => {
      toast.success(t('Ring Buffer Started.'))
      queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.error || error.message);
    },
  });
}

export interface PacketResponse {
  file_name: string;
  packets: Packet[];
  total: number;
}

export interface UsePacketParams {
  id: number;
  file_name: string;
  pageIndex: number;
  pageSize: number;
  sortBy: string;
  sortOrder: 'asc' | 'desc';
  filter?: string;
}

export function useViewPackets({
  id,
  file_name,
  pageIndex,
  pageSize,
  sortBy,
  sortOrder,
  filter,
}: UsePacketParams) {
  return useQuery<PacketResponse, AxiosError>({
    queryKey: [
      'wiresharkCapture',
      id,
      file_name,
      pageIndex,
      pageSize,
      sortBy,
      sortOrder,
      filter,
    ],
    queryFn: async () => {
      const response = await axios.get<PacketResponse>(
        `/api/v1/wireshark/ringbuffer/view`,
        {
          params: {
            id,
            file_name,
            page: pageIndex + 1,
            per_page: pageSize,
            sortBy,
            sortOrder,
            ...(filter ? { filter } : {}),
          },
        }
      );
      return response.data;
    },
    enabled: !!id && !!file_name,
  });
}


export interface TempFileResponse {
  message: string;
  id: number;
  merged_file: string;
  start_time: string | null;
  end_time: string | null;
}

export function useGetTempFile(id: number) {
  return useQuery<TempFileResponse, AxiosError>({
    queryKey: ['wiresharkTempFile', id],
    queryFn: async () => {
      const response = await axios.get<TempFileResponse>('/api/v1/wireshark/ringbuffer/tempfile', {
        params: { id },
      });
      return response.data;
    },
    enabled: !!id,
  });
}

export function useDownload(){
  const { t } = useTranslation();
  return useMutation({
    mutationFn: async (filePath: string) => {
      const response = await axios.post(
        '/api/v1/wireshark/ringbuffer/download',
        { file: filePath },
        { responseType: 'blob' } 
      );
      return response.data;
    },
    onSuccess:(data, filePath) => {
      const fileName = filePath.split(/[/\\]/).pop() || 'download.pcap';
      const url = window.URL.createObjectURL(new Blob([data]));
      const link = document.createElement('a');
      link.href = url;
      link.setAttribute('download', fileName); 
      document.body.appendChild(link);
      link.click();
      link.remove();
      toast.success(t('Ring Buffer Downloaded.'))
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.error || error.message);
    },
  });
}