import axios from "axios"
import { createContext, useContext } from "react"
import { User } from "@/components/user/types";
import { useQuery } from "@tanstack/react-query";
import { Trial, License } from "@/components/administration/license/types";

type AuthProviderState = {
  user: User | null;
  installed: boolean;
  isUserLoading: boolean;
  isInstalledLoading: boolean;
  trial: Trial | null;
  isTrialLoading: boolean;
  license: License | null;
  isLicenseLoading: boolean;
}

const initialState: AuthProviderState = {
  user: null,
  installed: false,
  isUserLoading: true,
  isInstalledLoading: true,
  trial: null,
  isTrialLoading: true,
  license: null,
  isLicenseLoading: true,
}

const AuthProviderContext = createContext<AuthProviderState>(initialState)

export function AuthProvider({
  children,
  ...props
}: { children: React.ReactNode }) {

  const { data: installed, isLoading: isInstalledLoading } = useQuery({
    queryKey: ['installed'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/auth/installed');
      return response.data.value == 1;
    },
    retry: false,
    refetchOnWindowFocus: false,
  })
 
  const { data: user, isLoading: isUserLoading } = useQuery({
    queryKey: ['auth'],
    queryFn: async () => {
      try {
        const response = await axios.get('/api/v1/auth');
        return response.data;
      } catch {
        return null;
      }
    },
    retry: false,
    refetchOnWindowFocus: false,
  })

  const { data: trial, isLoading: isTrialLoading } = useQuery({
    queryKey: ["trial"],
    queryFn: async () => {
      try {
        const response = await axios.get("/api/v1/administration/license/trial");
        return response.data;
      } catch {
        return null;
      }
    },
    retry: false,
    refetchInterval: (query) => {
      const trial = query.state.data as Trial | null;

      if (trial?.trial_time_left && trial.trial_time_left > 0) {
        return trial.trial_time_left * 1000;
      }

      return false;
    },
  });

  const { data: license, isLoading: isLicenseLoading } = useQuery({
    queryKey: ['license'],
    queryFn: async () => {
      try {
        const response = await axios.get("/api/v1/administration/license");
        return response.data;
      } catch {
        return null;
      }
    },
    retry: false,
    refetchOnWindowFocus: false,
  })

  return (
    <AuthProviderContext.Provider 
      {...props} 
      value={{ 
        user, 
        installed: installed ?? false, 
        isUserLoading, isInstalledLoading,
        trial,
        isTrialLoading,
        license,
        isLicenseLoading,
      }}
    >
      {children}
    </AuthProviderContext.Provider>
  )
}

export const useAuth = () => {
  const context = useContext(AuthProviderContext)

  if (context === undefined)
    throw new Error("useAuth must be used within an AuthProvider")

  return context
}
