import { useEffect, useState } from "react";
import { useTranslation } from "react-i18next";
import { Plus } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useForm } from "react-hook-form";
import { Trash2, Ellipsis } from "lucide-react";
import {
    Tooltip,
    TooltipContent,
    TooltipTrigger,
  } from "@/components/ui/tooltip"
import {
    DropdownMenu,
    DropdownMenuTrigger,
    DropdownMenuContent,
    DropdownMenuItem,
  } from "@/components/ui/dropdown-menu";
import {
    PageLayout,
    PageHeader,
    PageTitle,
    PageDescription,
    PageSeparator,
} from "@/components/Page";
import {
    Dialog,
    DialogContent,
    DialogHeader,
    DialogTitle,
    DialogDescription,
    DialogFooter,
} from "@/components/ui/dialog";
import {
  	Card,
  	CardContent,
  	CardTitle,
  	CardHeader,
	CardDescription,
} from "@/components/ui/card";
import {
  	Table,
  	TableBody,
  	TableCell,
  	TableHead,
  	TableHeader,
  	TableRow,
} from "@/components/ui/table";
import {
    Form,
    FormControl,
    FormField,
    FormItem,
    FormLabel,
} from "@/components/ui/form";
import { Textarea } from "@/components/ui/textarea";
import { Input } from "@/components/ui/input";
import AddEditAuthServerDialog from "@/components/administration/AddEditAuthServerDialog"
import { useGetAuthServers, useDeleteAuthServers, useGetCertificates, usePostCertificates, useDeleteCertificates, useExtractCN } from "@/components/administration/queries";
import { AuthServer } from "@/components/administration/types";
import ImportUserDialog from "@/components/administration/ImportUserDialog";
import { useQueryClient } from "@tanstack/react-query";

export default function AuthServers() {
    const { t } = useTranslation();
    const [open, setOpen] = useState(false);
    const { data: authServers, isLoading} = useGetAuthServers();
    const { mutate: deleteAuthServer } = useDeleteAuthServers();
    const queryClient = useQueryClient();
    // State for managing the edit dialog
    const [editDialogOpen, setEditDialogOpen] = useState(false);
    const [editingServer, setEditingServer] = useState<AuthServer | undefined>(undefined);
    
    const { data: certificates = [], isLoading: certsLoading } = useGetCertificates();
    const { mutate: addCertificate } = usePostCertificates();
    const { mutate: deleteCertificate } = useDeleteCertificates();
    
    //Form for the add certificate dialong
    const form = useForm({
        defaultValues: {
            hostname: "", 
            certificate: "",
        },
    });

    const cert = form.watch("certificate");

    const { mutateAsync: extractCN } = useExtractCN();
    
    useEffect(() => {
        const getCN = async () => {
            if (cert.includes("BEGIN CERTIFICATE")) {
                try {
                    const hostname = await extractCN(cert);
                    form.setValue("hostname", hostname || "");
                } catch {
                    form.setValue("hostname", "");
                }
            }
        };

        getCN();
    }, [cert, extractCN, form]);

    const onSubmit = (values: { certificate: string }) => {
        addCertificate(values, {
            onSuccess: () => {
                setOpen(false);
                form.reset();
            }
        });
    }

    return (
        <PageLayout>
            <PageHeader>
                <PageTitle>
                    {t("LDAP / Active Directory Integration")}
                </PageTitle>
                <PageDescription>
                    {t(
                        "Manage authentication servers can be used to authenticate users against during login. Once a server has been added you can"
                    )}{" "}
                    <ImportUserDialog onSuccess={() => queryClient.invalidateQueries({ queryKey: ['Users'] })}>
                        <span className="text-primary cursor-pointer">
                            {t("import users")}
                        </span>
                    </ImportUserDialog>.
                </PageDescription>
                <PageSeparator />
            </PageHeader>
            <Card>
                <CardTitle className="text-xl font-semibold px-6">
                    {t("LDAP / Active Directory Integration")}
                </CardTitle>
                <CardDescription className="px-6">
                    {t(
                        "Manage authentication servers can be used to authenticate users against during login. Once a server has been added you can"
                    )}{" "}
                    <ImportUserDialog onSuccess={() => queryClient.invalidateQueries({ queryKey: ['Users'] })}>
                        <span className="text-primary cursor-pointer">
                            {t("import users")}
                        </span>
                    </ImportUserDialog>.
                </CardDescription>
                <CardContent>
                    <AddEditAuthServerDialog>
                        <Button
                            type="button"
                            className="mb-2"
                            onClick={() => { 
                                    setEditingServer(undefined);
                                }}
                        >
                            <Plus />
                            {t("Add Server")}
                        </Button>
                    </AddEditAuthServerDialog>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead></TableHead>
                                <TableHead>{t("Name")}</TableHead>
                                <TableHead>{t("Server(s)")}</TableHead>
                                <TableHead>{t("Type")}</TableHead>
                                <TableHead>{t("Encryption")}</TableHead>
                                <TableHead>{t("Associated Users")}</TableHead>
                                <TableHead></TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {!isLoading && authServers && authServers.length > 0 ? (
                                authServers.map((AuthServer: { 
                                    id: number;
                                    active: boolean;
                                    name: string;
                                    host: string;
                                    type: string;
                                    basedn: string;
                                    encryption: string;
                                    suffix: string;
                                    controllers: string;
                                    port: number;
                                    associated_users: number; 
                                }) => (
                                    <TableRow key={AuthServer.id}>
                                        <TableCell className="w-4">
                                            <span
                                                className={`h-3 w-3 rounded-full inline-block ${
                                                    AuthServer.active ? "bg-success" : "bg-error"
                                                }`}
                                            />
                                        </TableCell>
                                        <TableCell>{AuthServer.name}</TableCell>
                                        <TableCell>
                                            {AuthServer.type.toLowerCase() === "ldap"
                                                ? AuthServer.host
                                                : AuthServer.controllers}
                                        </TableCell>
                                       <TableCell>
                                            {AuthServer.type === 'ldap'
                                                ? 'LDAP'
                                                : AuthServer.type === 'ad'
                                                ? 'Active Directory'
                                                : "Unknown"
                                            }
                                        </TableCell>
                                        <TableCell>
                                            {AuthServer.encryption === 'ssl' ? 'SSL/TLS' : AuthServer.encryption === 'tls' ? 'STARTTLS' : "None"}
                                        </TableCell>
                                        <TableCell>
                                            {AuthServer.associated_users ?? 0}
                                        </TableCell>
                                        <TableCell className="flex justify-end">
                                            <DropdownMenu>
                                                <DropdownMenuTrigger asChild>
                                                    <Button variant="ghost">
                                                        <Ellipsis />
                                                    </Button>
                                                </DropdownMenuTrigger>
                                                <DropdownMenuContent>
                                                    <DropdownMenuItem 
                                                        onClick={() => {
                                                            setEditingServer(AuthServer);
                                                            setEditDialogOpen(true);
                                                        }}
                                                    >
                                                        {t('Edit')}
                                                    </DropdownMenuItem>
                                                    {AuthServer.associated_users === 0 && (
                                                        <DropdownMenuItem onClick={() => deleteAuthServer(AuthServer.id)}>
                                                            {t('Delete')}
                                                        </DropdownMenuItem>
                                                    )}
                                                </DropdownMenuContent>
                                            </DropdownMenu>
                                        </TableCell>
                                    </TableRow>
                                ))
                            ) : (
                                <TableRow>
                                    <TableCell colSpan={6}>
                                        {t("No authentication servers have been added.")}
                                    </TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </CardContent>

                <CardHeader className="text-xl font-semibold px-6">
                    {t("Certificate Authority Management")}
                </CardHeader>
                <CardDescription className="px-6">
                    {t("If you are using self-signed certificates to connect over SSL/TLS, you will need to add the domain controllers' certificates to the local certificate authority.")}
                </CardDescription>
                <CardContent>
                    <Button
                        type="button"
                        className="mb-2"
                        onClick={() => setOpen(true)}
                    >
                        <Plus/>
                        {t("Add Certificate")}
                    </Button>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>{t("Hostname")}</TableHead>
                                <TableHead>{t("Issuer (CA)")}</TableHead>
                                <TableHead>{t("Expiration Date")}</TableHead>
                                <TableHead></TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {!certsLoading && certificates.length > 0 ? (
                                certificates.map((cert: { id: number; subject_cn: string; issuer_cn: string; valid_to: number }) => (
                                    <TableRow key={cert.id}>
                                        <TableCell>{cert.subject_cn}</TableCell>
                                        <TableCell>{cert.issuer_cn}</TableCell>
                                        <TableCell>{new Date(cert.valid_to * 1000).toString()}</TableCell>
                                        <TableCell className="flex justify-end pr-5">
                                            <Tooltip>
                                                <TooltipTrigger asChild>
                                                <Trash2
                                                    onClick={() => deleteCertificate(cert.id)}
                                                    className="w-4 h-5 cursor-pointer"
                                                />
                                                </TooltipTrigger>
                                                <TooltipContent>
                                                    <p>{t('Delete')}</p>
                                                </TooltipContent>
                                            </Tooltip>
                                        </TableCell>
                                    </TableRow>
                                ))
                            ) : (
                                <TableRow>
                                    <TableCell colSpan={4}>
                                        {t("No certificates have been added.")}
                                    </TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
            
            {/* DIALOG FOR CERTS*/}
            <Dialog open={open} onOpenChange={setOpen}>
                <DialogContent className="sm:max-w-[600px]">
                    <DialogHeader>
                        <DialogTitle>
                            {t("Add Certificate")}
                        </DialogTitle>
                        <DialogDescription>
                            {t("To add a certificate to the certificate authority, copy and paste the actual certificate between, and including, the begin/end certificate sections.")}
                        </DialogDescription>
                    </DialogHeader>
                    <Form {...form}>
                        <form
                            onSubmit={form.handleSubmit(onSubmit)}
                            className="grid gap-4"
                        >
                            <FormField
                                control={form.control}
                                name="hostname"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>
                                            {t("Hostname")}
                                        </FormLabel>
                                        <FormControl>
                                            <Input
                                                {...field}
                                                disabled
                                                className="w-full bg-muted text-muted-foreground"
                                            />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />

                            <FormField
                            control={form.control}
                            name="certificate"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>
                                        {t("Certificate")}
                                    </FormLabel>
                                    <FormControl>
                                        <Textarea
                                            {...field}
                                            className="h-60 font-mono text-sm resize-y"
                                            placeholder="Enter your certificate here"
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                            />
                            <DialogFooter>
                                <Button
                                    type="button"
                                    variant="secondary"
                                    onClick={() => setOpen(false)}
                                >
                                    {t("Close")}
                                </Button>
                                <Button type="submit">
                                    {t("Add")}
                                </Button>
                            </DialogFooter>
                        </form>
                    </Form>
                </DialogContent>
            </Dialog>

            {/* EDIT AUTH SERVER DIALOG - Rendered outside the dropdown */}
            {editingServer && (
                <AddEditAuthServerDialog 
                    AuthServer={editingServer}
                    open={editDialogOpen}
                    onOpenChange={(open) => {
                        setEditDialogOpen(open);
                        if (!open) {
                            setEditingServer(undefined);
                        }
                    }}
                />
            )}
        </PageLayout>
    );
}