import { useTranslation } from "react-i18next";
import {
  Ellipsis,
  Trash2,
  Download,
} from "lucide-react"
import { useEcho } from '@laravel/echo-react';
import { useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import {
  useCreateBackup,
  useGetBackups,
  useDeleteBackup,
  useDownloadBackup
} from "@/components/administration/queries";
import { BaseBackup } from "@/components/administration/types";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { ColumnDef } from "@tanstack/react-table";
import {
    PageLayout,
    PageHeader,
    PageTitle,
    PageDescription,
    PageSeparator,
} from "@/components/Page";
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";

export default function Backup() {
  const { t } = useTranslation();
  const { data, isLoading } = useGetBackups();
  const createBackup = useCreateBackup();
  const deleteBackup = useDeleteBackup();
  const downloadBackup = useDownloadBackup();
  const queryClient = useQueryClient();

  useEcho('NNA.Admins', 'BackupCreated', () => {
    queryClient.invalidateQueries({ queryKey: ["backups"] });
  });

  const columns: ColumnDef<BaseBackup>[] = [
    {
      accessorKey: 'file_name',
      id: "File Name",
      header: t('File Name'),
    },
    {
      accessorKey: 'date',
      header: t('Date'),
    },
    {
      accessorKey: 'size',
      header: t('Size'),
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => (
        <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost">
            <Ellipsis />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent>
          <DropdownMenuItem onClick={() => downloadBackup.mutate(row.original.file_name)}>
            <Download />
            {t('Download')}
          </DropdownMenuItem>
          <DropdownMenuItem onClick={() => deleteBackup.mutate(row.original.file_name)}>
            <Trash2 />
            {t('Delete')}
          </DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>
      ),
    },
  ];

  return (
      <PageLayout>
        <PageHeader>
          <PageTitle>{t('System Backup')}</PageTitle>
          <PageDescription>{t('Manage your Nagios Network Analyzer backups')}</PageDescription>
          <Button
            onClick={() => {
              createBackup.mutate();
            }}
          >
            {t('Create Backup')}
          </Button>
          <PageSeparator />
      </PageHeader>
      {isLoading && (
          <DataTableSkeleton></DataTableSkeleton>
      )}
      {!isLoading && (
          <DataTable
            columns={columns}
            data={data || []}
          />
                  )}
    </PageLayout>
  );
}
