import { useTranslation } from "react-i18next";
import { useForm } from "react-hook-form";
import { Input } from "@/components/ui/input";
import { useNavigate } from 'react-router-dom';
import ThemesDropdown from "@/components/ui/ThemesDropdown";
import { RadioGroup } from '@radix-ui/react-radio-group';
import { RadioGroupItem } from '@/components/ui/radio-group';
import { Button } from "@/components/ui/button";
import { Label } from '@/components/ui/label';
import { PasswordInput } from '@/components/ui/passwordInput';
import { useCreateUser } from "@/components/user/queries";
import type { Language, User } from "@/components/user/types";
import { Select, SelectTrigger, SelectValue, SelectContent, SelectGroup, SelectItem } from "@/components/ui/select";
import { useGetAuthServers } from "@/components/administration/queries";
import { AuthServer } from "@/components/administration/types";
import { useEffect } from "react";
import {
    Command,
    CommandEmpty,
    CommandGroup,
    CommandInput,
    CommandItem,
    CommandList,
} from "@/components/ui/command";
import {
    Popover,
    PopoverContent,
    PopoverTrigger,
} from "@/components/ui/popover";
import {
    PageLayout,
    PageHeader,
    PageTitle,
    PageDescription,
    PageSeparator,
    PageSectionDescription,
} from "@/components/Page";
  import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"
import {
    Card,
    CardContent,
    CardFooter,
} from "@/components/ui/card";
import React, { useState } from "react";
import { cn } from "@/lib/utils";
import { Check, ChevronsUpDown } from "lucide-react";

export default function CreateUser() {
    const { t } = useTranslation();
    const navigate = useNavigate();
    const [open, setOpen] = useState(false);
    const [authType, setAuthType] = useState<'local' | 'ldap' | 'ad'>('local');
    const { data: authServers } = useGetAuthServers();
    const ldapServers = (authServers ?? []).filter((server: AuthServer) => server.type === "ldap");
    const adServers = (authServers ?? []).filter((server: AuthServer) => server.type === "ad");

    type FormNames = "first_name" | "last_name" | "company" | "email" | "phone" | "id" | "auth" | "username" | "password" | "lang" | "theme" | "is_admin" | "apiaccess" | "default_dashboard";
    const userDetails: { name: FormNames; label: string, description: string, placeholder?: string }[] = [
        { name: "company", label: t('Company'), description: t('Name displayed in Nagios Applications'), placeholder: t('Nagios Enterprises') },
        { name: "email", label: t('Email'), description: t('Your email address for account access'), placeholder: 'example@domain.com' },
        { name: "phone", label: t('Phone Number'), description: t('Contact phone number'), placeholder: '+1 123-456-7890' },
    ];

    const languages: { label: string; value: Language }[] = [
        { label: t("Bulgarian"), value: "bg_BG" },
        { label: t("Czech"), value: "cs_CZ" },
        { label: t("German"), value: "de_DE" },
        { label: t("English"), value: "en_US" },
        { label: t("Spanish"), value: "es_ES" },
        { label: t("French"), value: "fr_FR" },
        { label: t("Italian"), value: "it_IT" },
        { label: t("Japanese"), value: "ja_JP" },
        { label: t("Korean"), value: "ko_KR" },
        { label: t("Polish"), value: "pl_PL" },
        { label: t("Portugese"), value: "pt_PT" },
        { label: t("Russian"), value: "ru_RU" },
        { label: t("Chinese (China)"), value: "zh_CN" },
        { label: t("Chinese (Taiwan)"), value: "zh_TW" },
    ];

    const form = useForm<User & { confirm_password: string }>({
        defaultValues: {
            id: 0,
            auth: "local",
            first_name: "",
            last_name: "",
            company: "",
            phone: "",
            username: "",
            password: "",
            confirm_password: "",
            email: "",
            lang: "en_US",
            theme: "Default",
            is_admin: "0",
            apiaccess: "0",
            default_dashboard: 0,
            auth_server_id: 0,
            auth_server_data: "",
        }
    });

    const { isDirty } = form.formState;
    const createUser = useCreateUser();

    useEffect(() => {
        if (authType === "ad") {
            if (adServers.length > 0) {
              form.setValue("auth_server_id", adServers[0].id, { shouldDirty: true });
            } else {
              form.setValue("auth_server_id", 0, { shouldDirty: true });
            }
        } else if (authType === "ldap") {
            if (ldapServers.length > 0) {
              form.setValue("auth_server_id", ldapServers[0].id, { shouldDirty: true });
            } else {
              form.setValue("auth_server_id", 0, { shouldDirty: true });
            }
        } else {
            form.setValue("auth_server_id", 0, { shouldDirty: true });
        }
      }, [authType, adServers, ldapServers]);

    function onSubmit(values: User) {
        createUser.mutate(values);
    }
    return (
        <PageLayout>
            <PageHeader>
                <PageTitle>{t('Create User')}</PageTitle>
                    <PageDescription>{t('Please enter all fields below')}</PageDescription>
                <PageSeparator />
            </PageHeader>
            
            <Form {...form}>
                <form onSubmit={form.handleSubmit((onSubmit))} className="space-y-6">
                    <Card>
                        <CardContent>
                            <div className="flex flex-row items-center justify-between">
                                <h1 className="leading-9 text-md font-large font-semibold">
                                    {t("User Details")}
                                </h1>
                                <div className="space-x-4">
                                    <Button
                                        type="button"
                                        variant={"destructive"}
                                        onClick={() => navigate("/admin/users")}
                                    >
                                        {t("Cancel")}
                                    </Button>
                                    <Button
                                        disabled={!isDirty}
                                        variant={isDirty ? "default" : "outline"}
                                        className="border"
                                        type="submit"
                                    >
                                        {t("Create User")}
                                    </Button>
                                </div>
                            </div>
                            <PageSeparator />
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                <div className="space-y-1">
                                    <FormLabel>{t("Full Name")}</FormLabel>
                                    <FormDescription>
                                        {t("Name displayed in interface")}
                                    </FormDescription>
                                </div>
                                <div className="flex gap-2 w-md">
                                    <FormField
                                        control={form.control}
                                        name="first_name"
                                        render={({ field }) => (
                                            <FormItem className="w-full">
                                                <FormControl>
                                                    <Input placeholder={t("First Name")} {...field} value={field.value} />
                                                </FormControl>
                                                <FormMessage />
                                            </FormItem>
                                        )}
                                    />
                                    <FormField
                                        control={form.control}
                                        name="last_name"
                                        render={({ field }) => (
                                            <FormItem className="w-full">
                                                <FormControl>
                                                    <Input placeholder={t("Last Name")} {...field} value={field.value} />
                                                </FormControl>
                                                <FormMessage />
                                            </FormItem>
                                        )}
                                    />
                                </div>
                            </FormItem>
                            <PageSeparator />
                            {userDetails.map(({ name, label, description, placeholder }, index) => (
                                <React.Fragment key={name}>
                                    <FormField 
                                        control={form.control}
                                        name={name}
                                        render={({ field }) => (
                                            <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                                <div className="space-y-1">
                                                    <FormLabel>{label}</FormLabel>
                                                    <FormDescription>{description}</FormDescription>
                                                </div>
                                                <FormControl className="w-md">
                                                    <Input {...field} placeholder={placeholder} />
                                                </FormControl>
                                                <FormMessage />
                                            </FormItem>
                                        )}
                                    />
                                    {index !== userDetails.length - 1 && <PageSeparator />}
                                </React.Fragment>
                            ))}
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent>
                            <h1 className="leading-9 text-md font-large font-semibold">
                                {t("Account Information")}
                            </h1>
                            <PageSeparator />
                            <FormField 
                                control={form.control}
                                name="username"
                                render={({ field }) => (
                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <div className="space-y-1">
                                            <FormLabel>{t('Username')}</FormLabel>
                                            <FormDescription>{t("Your unique identifier in Network Analyzer")}</FormDescription>
                                        </div>
                                        <FormControl className="w-md">
                                            <Input {...field} placeholder='nagiosadmin' />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <PageSeparator />
                            {authType === "local" && (
                            <>
                                <FormField 
                                    control={form.control}
                                    name="password"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Password')}</FormLabel>
                                                <FormDescription>{t('Your secure password for authentication')}</FormDescription>
                                            </div>
                                            <FormControl>
                                                <div className="w-md">
                                                    <PasswordInput {...field} />
                                                </div>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                                <PageSeparator />
                                <FormField 
                                    control={form.control}
                                    name="confirm_password"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                            <div className="space-y-1">
                                                <FormLabel>{t('Confirm Password')}</FormLabel>
                                                <FormDescription>{t('Re-enter your password')}</FormDescription>
                                            </div>
                                            <FormControl className="w-md">
                                                <div className="w-md">
                                                    <PasswordInput {...field} />
                                                </div>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                                <PageSeparator />
                            </>
                            )}
                            <FormField
                                control={form.control}
                                name="lang"
                                render={({ field }) => (
                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <div className="space-y-1">
                                            <FormLabel>{t("Language")}</FormLabel>
                                            <FormDescription>
                                                {t("Set the display language")}
                                            </FormDescription>
                                        </div>
                                        <Popover open={open} onOpenChange={setOpen}>
                                            <PopoverTrigger asChild>
                                                <FormControl>
                                                    <Button
                                                        variant="outline"
                                                        role="combobox"
                                                        className={cn(
                                                            "w-[200px] justify-between",
                                                            !field.value &&
                                                                "text-muted-foreground"
                                                        )}
                                                    >
                                                        {field.value
                                                            ? languages.find(
                                                                (lang) =>
                                                                    lang.value ===
                                                                    field.value
                                                            )?.label
                                                            : "Select language"}
                                                        <ChevronsUpDown className="opacity-50" />
                                                    </Button>
                                                </FormControl>
                                            </PopoverTrigger>
                                            <PopoverContent className="w-[200px] p-0">
                                                <Command>
                                                    <CommandInput placeholder="Search language..." />
                                                    <CommandList>
                                                        <CommandEmpty>
                                                            {t("No language found")}
                                                        </CommandEmpty>
                                                        <CommandGroup>
                                                            {languages.map(
                                                                (language) => (
                                                                    <CommandItem
                                                                        value={language.label}
                                                                        key={language.value}
                                                                        onSelect={() => {
                                                                            setOpen(false);
                                                                            form.setValue("lang", language.value);
                                                                          }}
                                                                    >
                                                                        {language.label}
                                                                        <Check
                                                                            className={cn(
                                                                                "ml-auto",
                                                                                language.value ===
                                                                                    field.value
                                                                                    ? "opacity-100"
                                                                                    : "opacity-0"
                                                                            )}
                                                                        />
                                                                    </CommandItem>
                                                                )
                                                            )}
                                                        </CommandGroup>
                                                    </CommandList>
                                                </Command>
                                            </PopoverContent>
                                        </Popover>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <PageSeparator />
                            <FormField
                                control={form.control}
                                name="theme"
                                render={({ field }) => (
                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <div className="space-y-1">
                                            <FormLabel>{t('User Theme')}</FormLabel>
                                            <FormDescription>{t('Set the display theme')}</FormDescription>
                                        </div>
                                        <FormControl>
                                            <div className="w-[200px]">
                                                <ThemesDropdown field={field} />
                                            </div>
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent>
                            <h1 className="leading-9 text-md font-large font-semibold">
                                {t("Authentication Settings")}
                            </h1>
                            <PageSectionDescription>{t('User accounts can be authenticated in many different ways either from your local database or external programs such as Active Directory or LDAP. You can set up external authentication servers in the LDAP/AD Integration settings.')}</PageSectionDescription>
                            <PageSeparator />
                            <FormField
                                control={form.control}
                                name="auth"
                                render={({ field }) => (
                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <div className="space-y-1">
                                            <FormLabel>{t("Authentication Type")}</FormLabel>
                                            <FormDescription>
                                            {t("Choose how this user will authenticate.")}
                                            </FormDescription>
                                        </div>
                                        <FormControl className="w-md">
                                            <div className="col-span-2 w-full">
                                                <Select
                                                    value={field.value}
                                                    onValueChange={(selectedVal) => {
                                                        setAuthType(selectedVal as 'local' | 'ldap' | 'ad'); // update your local state if needed
                                                        field.onChange(selectedVal); // update form value
                                                    }}
                                                    >
                                                    <SelectTrigger className="w-full">
                                                        <SelectValue placeholder={t("Select authentication type")} />
                                                    </SelectTrigger>
                                                    <SelectContent>
                                                        <SelectGroup>
                                                        <SelectItem value="local">{t("Local")}</SelectItem>
                                                        <SelectItem value="ad" disabled={adServers.length === 0}>{t("Active Directory")}</SelectItem>
                                                        <SelectItem value="ldap" disabled={ldapServers.length === 0}>{t("LDAP")}</SelectItem>
                                                        </SelectGroup>
                                                    </SelectContent>
                                                </Select>
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                                />
                                {(authType === "ldap" || authType === "ad") && (
                                <>
                                <PageSeparator />
                                    <FormField
                                    control={form.control}
                                    name="auth_server_id"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <FormLabel>
                                            {authType === "ldap" ? t('LDAP Server') : t('AD Server')}
                                        </FormLabel>
                                        <FormControl className="w-md">
                                            <div className="col-span-2 w-full">
                                            <Select
                                                value={field.value?.toString()}
                                                onValueChange={(selectedVal) => {
                                                field.onChange(selectedVal)
                                            }}>
                                                <SelectTrigger className="w-full">
                                                    <SelectValue placeholder="Select a server" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                <SelectGroup className="w-full max-h-80">
                                                {(authType === "ldap" ? ldapServers : adServers).map((server: AuthServer) => (
                                                        <SelectItem key={server.id} value={server.id.toString()}>
                                                            {server.name} ({authType === "ldap" ? server.host : server.controllers})
                                                        </SelectItem>
                                                    ))}
                                                </SelectGroup>
                                            </SelectContent>
                                            </Select>
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                        </FormItem>
                                    )}
                                    />
                                    <PageSeparator />
                                    <FormField
                                    control={form.control}
                                    name="auth_server_data"
                                    render={({ field }) => (
                                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <FormLabel>
                                            {authType === "ldap"
                                            ? t("User's Full DN")
                                            : t("AD Username")}
                                        </FormLabel>
                                        <FormControl className="w-md">
                                            <Input
                                            {...field}
                                            placeholder={
                                                authType === "ldap"
                                                ? "cn=John Smith,dn=nagios,dc=com"
                                                : t("AD Username")
                                            }
                                            />
                                        </FormControl>
                                        <FormMessage />
                                        </FormItem>
                                    )}
                                    />
                                </>
                                )}
                            <PageSeparator />
                            <FormField 
                                control={form.control}
                                name="is_admin"
                                render={({ field }) => (
                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <div className="space-y-1">
                                            <FormLabel>{t('User Level Access')}</FormLabel>
                                            <FormDescription>{t("Set the user level of access inside the UI. Admin's have full access while User's can view components but only edit their own profile information.")}</FormDescription>
                                        </div>
                                        <FormControl className="w-md">
                                            <RadioGroup
                                                className="gap-4 grid"
                                                onValueChange={field.onChange}  
                                                value={field.value}
                                            >
                                                <div className="flex items-center space-x-2">
                                                    <RadioGroupItem value="1" id="user-option-one" />
                                                    <Label htmlFor="user-option-one">{t('Admin')}</Label>
                                                </div>
                                                <div className="flex items-center space-x-2">
                                                    <RadioGroupItem value="0" id="user-option-two" />
                                                    <Label htmlFor="user-option-two">{t('User')}</Label>
                                                </div>
                                            </RadioGroup>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <PageSeparator />
                            <FormField 
                                control={form.control}
                                name="apiaccess"
                                render={({ field }) => (
                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                        <div className="space-y-1">
                                            <FormLabel>{t('API Access')}</FormLabel>
                                            <FormDescription>{t('If you want to allow this user to use the external API via an access key')}</FormDescription>
                                        </div>
                                        <FormControl className="w-md">
                                            <RadioGroup
                                                defaultValue="option-one"
                                                className="gap-4 grid"
                                                onValueChange={field.onChange}  
                                                value={field.value}
                                            >
                                                <div className="flex items-center space-x-2">
                                                    <RadioGroupItem value="1" id="api-option-one" />
                                                    <Label htmlFor="api-option-one">{t('Yes')}</Label>
                                                </div>
                                                <div className="flex items-center space-x-2">
                                                    <RadioGroupItem value="0" id="api-option-two" />
                                                    <Label htmlFor="api-option-two">{t('No')}</Label>
                                                </div>
                                            </RadioGroup>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </CardContent>
                        <CardFooter className="justify-end space-x-4">
                            <Button
                                type="button"
                                variant={"destructive"}
                                onClick={() => navigate("/admin/users")}
                            >
                                {t("Cancel")}
                            </Button>
                            <Button
                                disabled={!isDirty}
                                variant={isDirty ? "default" : "outline"}
                                className="border"
                                type="submit"
                            >
                                {t("Create User")}
                            </Button>
                        </CardFooter>
                    </Card>
                </form>
            </Form>
        </PageLayout>
    );
  }