import { Button } from "@/components/ui/button";
import { Ellipsis, Plus, Pencil, Trash2 } from "lucide-react";
import { useTranslation } from "react-i18next";
import {
  PageLayout,
  PageHeader,
  PageTitle,
  PageSeparator,
} from "@/components/Page";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { ColumnDef } from "@tanstack/react-table";
import { BaseUser, User } from "@/components/user/types";
import { useGetUsers } from "@/components/user/queries";
import { useDeleteUser } from "@/components/user/queries";
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";
import ImportUserDialog from "@/components/administration/ImportUserDialog";
import AddEditUserDialog from "@/components/user/AddEditUserDialog";
import { useQueryClient } from "@tanstack/react-query";
import { useState } from "react";

export default function Users() {
  const { t } = useTranslation();
  const { data, isLoading } = useGetUsers();
  const queryClient = useQueryClient();
  const deleteUserMutation = useDeleteUser();

  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [editingUser, setEditingUser] = useState<User | undefined>(undefined);

  // Define columns for the DataTable
  const columns: ColumnDef<BaseUser>[] = [
    {
      accessorKey: 'username',
      header: t('Username'),
    },
    {
      accessorKey: 'email',
      header: t('Email'),
    },
    {
      id: 'is_admin',
      header: t('Access Level'),
      cell: ({ row }) => (
        <span>{row.original.is_admin ? t('Admin') : t('User')}</span>
      ),
    },
    {
      accessorKey: 'type',
      header: t('Account Type'),
    },
    {
      accessorKey: 'apiaccess',
      header: t('API Access'),
      cell: ({ row }) => (
        <span>{row.original.apiaccess ? t('Yes') : t('No')}</span>
      ),
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => (
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <Button variant="ghost">
              <Ellipsis />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent>
            <DropdownMenuItem 
              onClick={() => {
                setEditingUser(row.original as User);
                setEditDialogOpen(true);
              }}
            >
              <Pencil />
              {t('Edit')}
            </DropdownMenuItem>
            <DropdownMenuItem onClick={() => deleteUserMutation.mutate(row.original.id)}>
              <Trash2 />
              {t('Delete')}
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      ),
    },
  ];

  return (
    <PageLayout>
      <PageHeader>
        <PageTitle>{t('User Management')}</PageTitle>
        <div className="gap-4 flex">
          <AddEditUserDialog>
            <Button onClick={() => setEditingUser(undefined)}>
              <Plus />
              {t('Create User')}
            </Button>
          </AddEditUserDialog>
          
          <ImportUserDialog onSuccess={() => queryClient.invalidateQueries({ queryKey: ['Users'] })}>
            <Button> <Plus />{t('Add Users from LDAP/AD')}</Button>
          </ImportUserDialog>
        </div>
        <PageSeparator />
      </PageHeader>
      
      {isLoading && (
        <DataTableSkeleton></DataTableSkeleton>
      )}
      {!isLoading && (
        <DataTable columns={columns} data={data || []} />
      )}

      {/* Edit User Dialog - Rendered outside the dropdown */}
      {editingUser && (
        <AddEditUserDialog 
          user={editingUser}
          open={editDialogOpen}
          onOpenChange={(open) => {
            setEditDialogOpen(open);
            if (!open) {
              setEditingUser(undefined);
            }
          }}
        />
      )}
    </PageLayout>
  );
}