
import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Row } from "@tanstack/react-table";
import { useTranslation } from "react-i18next";
import { DropdownMenu, DropdownMenuItem, DropdownMenuContent, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import {
  ActivityIcon,
  HardDriveIcon,
  Loader2Icon,
  MoreHorizontalIcon,
  SquareTerminalIcon,
  TargetIcon,
  Trash2Icon,
  Settings2Icon,
} from "lucide-react";
import AddNagiosServerModal from "@/components/alerting/modals/settings/AddNagiosServer";
import AddServiceHostnamesModal from "@/components/alerting/modals/settings/AddServiceHostname";
import {
  useDeleteCommand,
  useDeleteServiceHostname,
  useDeleteNagiosServer,
  useGetCommands,
  useGetNagiosServers,
  useGetServiceHostnames,
  useGetSNMPReceivers,
  useDeleteSNMPReceiver
} from "@/components/alerting/queries";
import AddSNMPReceiverModal from "@/components/alerting/modals/settings/AddSnmpReceiver";
import AddCommandModal from "@/components/alerting/modals/settings/AddCommand";
import { Command, NagiosServer, ServiceHostname, SNMPReceiver } from "@/components/alerting/types";
import { useState } from "react";
import EditCommandModal from "@/components/alerting/modals/settings/EditCommand";
import EditNagiosServerModal from "@/components/alerting/modals/settings/EditNagiosServer";
import EditServiceHostnameModal from "@/components/alerting/modals/settings/EditServiceHostname";
import EditSNMPReceiverModal from "@/components/alerting/modals/settings/EditSnmpReceiver";
import { type TFunction } from "i18next";
import { useAuth } from "@/contexts/AuthContext";

import {
  Sheet,
  SheetContent,
  SheetDescription,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from "@/components/ui/sheet"  

export default function AlertingResponseSettings() {
  const { t } = useTranslation();
  const [sheetOpen, setSheetOpen] = useState(false);

  return (

    <div>
      <Sheet open={sheetOpen} onOpenChange={setSheetOpen}>
        <SheetTrigger asChild>
          <Button variant={"outline"} className="w-fit">
            <Settings2Icon size={16} />
            {t('Alerting Configs')}
          </Button>
        </SheetTrigger>
        <SheetContent className="sm:max-w-[800px] overflow-y-auto custom-scrollbar">
          <SheetHeader>
            <SheetTitle>{`Alerting Configs`}</SheetTitle>
            <SheetDescription>
              {`Configure how alerts are handled and what actions are taken.`}
            </SheetDescription>
          </SheetHeader>
          <div className="mt-4">
            <Tabs defaultValue="nagios-setup">
              <TabsList className="mb-4 w-min">
                <TabsTrigger value="nagios-setup" className="flex items-center gap-1.5">
                  <HardDriveIcon size={16} />
                  {t('Nagios Setup')}
                </TabsTrigger>
                <TabsTrigger value="snmp-receivers" className="flex items-center gap-1.5">
                  <TargetIcon size={16} />
                  {t('SNMP Receivers')}
                </TabsTrigger>
                <TabsTrigger value="commands" className="flex items-center gap-1.5">
                  <SquareTerminalIcon size={16} />
                  {t('Commands')}
                </TabsTrigger>
              </TabsList>

              <TabsContent value="nagios-setup">
                <AlertingNagiosSetup t={t} />
              </TabsContent>

              <TabsContent value="snmp-receivers">
                <AlertingSnmpReceivers t={t} />
              </TabsContent>

              <TabsContent value="commands">
                <AlertingCommands t={t} />
              </TabsContent>
            </Tabs>
          </div>
        </SheetContent>
      </Sheet>

    </div>
  );
}


const AlertingNagiosSetup = ({ t }: { t: TFunction }) => {
  const [nagiosServerDropdownOpen, setNagiosServerDropdownOpen] = useState(false);
  const { user } = useAuth();

  const { data: nagiosServers, isLoading: nagiosServersLoading } = useGetNagiosServers();
  const { data: serviceHostnames, isLoading: serviceHostnamesLoading } = useGetServiceHostnames();

  const { mutate: deleteNagiosServer } = useDeleteNagiosServer();
  const { mutate: deleteServiceHostname } = useDeleteServiceHostname();

  const alertingNagiosSetupColumns = [
    {
      accessorKey: 'name',
      header: 'Name',
    },
    {
      accessorKey: 'nrdp_url',
      header: 'NRDP URL',
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => (
        <div className="hidden" />
      ),
      cell: ({ row }: { row: Row<any> }) => {
        return (
          <div className="flex justify-end">
            {!!user?.is_admin &&
              <DropdownMenu open={nagiosServerDropdownOpen} onOpenChange={setNagiosServerDropdownOpen}>
                <DropdownMenuTrigger asChild>
                  <Button size="sm" variant="ghost" className="gap-2 text-xs">
                    <MoreHorizontalIcon />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  <EditNagiosServerModal nagiosServer={nagiosServers.filter((nagiosServer: NagiosServer) => nagiosServer.id == row.original.id)[0]} onModalClose={() => setNagiosServerDropdownOpen(false)} />
                  <DropdownMenuItem onClick={() => deleteNagiosServer(row.original.id)}>
                    <Trash2Icon />
                    <span className="hidden lg:block">{t(`Delete`)}</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            }
          </div>
        )
      },
    },
  ];

  const alertingServiceHostnamesColumns = [
    {
      accessorKey: 'servicename',
      header: 'Service Name',
    },
    {
      accessorKey: 'hostname',
      header: 'Hostname',
    },
    {
      accessorKey: 'nagios_server_id',
      header: 'Nagios Server',
      cell: ({ row }: { row: Row<any> }) => (
        <div className="">
          <span className="text-xs">{nagiosServers?.find((nagiosServer: NagiosServer) => nagiosServer.id === row.original.server_id)?.name}</span>
        </div>
      ),
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => (
        <div className="hidden" />
      ),
      cell: ({ row }: { row: Row<any> }) => {
        const [serviceHostnameDropdownOpen, setServiceHostnameDropdownOpen] = useState(false);

        return (
          <div className="flex justify-end">
            {!!user?.is_admin &&
              <DropdownMenu open={serviceHostnameDropdownOpen} onOpenChange={setServiceHostnameDropdownOpen}>
                <DropdownMenuTrigger asChild>
                  <Button size="sm" variant="ghost" className="gap-2 text-xs">
                    <MoreHorizontalIcon />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  <EditServiceHostnameModal serviceHostname={serviceHostnames.filter((serviceHostname: ServiceHostname) => serviceHostname.id == row.original.id)[0]} onModalClose={() => setServiceHostnameDropdownOpen(false)} />
                  <DropdownMenuItem onClick={() => deleteServiceHostname(row.original.id)}>
                    <Trash2Icon />
                    <span className="hidden lg:block">{t(`Delete`)}</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            }
          </div>
        )
      },
    }
  ];

  return (
    <div className="flex flex-col gap-12">
      {/* Service/Hostnames Section */}
      <div className="flex flex-col gap-4">
        <div className="flex gap-2 w-full justify-between">
          <div className="flex flex-col gap-1">
            <h1 className="text-xl font-medium flex items-center gap-1.5">
              <ActivityIcon size={20} />
              {t(`Service/Hostnames`)}
            </h1>
            <h3 className="text-sm text-muted-foreground">
              {t(`Define the service/hostnames sent to Nagios XI, appearing as unconfigured objects until configured in Nagios XI.`)}
            </h3>
          </div>
          <AddServiceHostnamesModal />
        </div>
        {
          serviceHostnamesLoading ? (
            <div className="flex justify-center items-center h-48">
              <Loader2Icon className="animate-spin" />
            </div>
          ) : (
            serviceHostnames && <DataTable columns={alertingServiceHostnamesColumns} data={serviceHostnames || []} hideView />
          )
        }
      </div>
      {/* Nagios Servers Section */}
      <div className="flex flex-col gap-4">
        <div className="flex gap-2 w-full justify-between">
          <div className="flex flex-col gap-1">
            <h1 className="text-xl font-medium flex items-center gap-1.5">
              <HardDriveIcon size={20} />
              {t(`Nagios Servers`)}
            </h1>
            <h3 className="text-sm text-muted-foreground">
              {t(`Define the Nagios XI servers where alerts/notifications will be sent.`)}
            </h3>
          </div>
          <AddNagiosServerModal />
        </div>
        {
          nagiosServersLoading ? (
            <div className="flex justify-center h-48">
              <Loader2Icon className="animate-spin" />
            </div>
          ) : (
            nagiosServers && <DataTable columns={alertingNagiosSetupColumns} data={nagiosServers || []} hideView hidePageSizeSelector />
          )
        }
      </div>
    </div>
  );
};

const AlertingSnmpReceivers = ({ t }: { t: TFunction }) => {
  const [snmpReceiverDropdownOpen, setSnmpReceiverDropdownOpen] = useState(false);
  const { data: snmp_receivers, isLoading: snmp_receivers_loading } = useGetSNMPReceivers();
  const { mutate: deleteSNMPReceiver } = useDeleteSNMPReceiver();
  const { user } = useAuth();

  const alertingSnmpReceiversColumns = [
    {
      accessorKey: 'name',
      header: 'Name',
    },
    {
      accessorKey: 'ip',
      header: 'IP Address',
    },
    {
      accessorKey: 'port',
      header: 'Port',
    },
    {
      accessorKey: 'version',
      header: 'Version',
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => (
        <div className="hidden" />
      ),
      cell: ({ row }: { row: Row<any> }) => {
        return (
          <div className="flex justify-end">
            {!!user?.is_admin &&   
              <DropdownMenu open={snmpReceiverDropdownOpen} onOpenChange={setSnmpReceiverDropdownOpen}>
                <DropdownMenuTrigger asChild>
                  <Button size="sm" variant="ghost" className="gap-2 text-xs">
                    <MoreHorizontalIcon />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  {
                    snmp_receivers && (
                      <EditSNMPReceiverModal snmp_receiver={snmp_receivers.filter((snmp_receiver: SNMPReceiver) => snmp_receiver.id == row.original.id)[0]} onModalClose={() => setSnmpReceiverDropdownOpen(false)} />
                    )
                  }
                  <DropdownMenuItem onClick={() => deleteSNMPReceiver(row.original.id)}>
                    <Trash2Icon />
                    <span className="hidden lg:block">{t(`Delete`)}</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            }
          </div>
        )
      }
    },
  ];

  return (
    <div className="flex flex-col gap-4">
      <div className="flex gap-2 w-full justify-between">
        <div className="flex flex-col gap-1">
          <h1 className="text-xl font-medium flex items-center gap-1.5">
            <TargetIcon size={20} />
            {t(`SNMP Receivers`)}
          </h1>
          <h3 className="text-sm text-muted-foreground">
            {t(`Define the SNMP receivers where alerts/notifications will be sent.`)}
          </h3>
        </div>
        <AddSNMPReceiverModal />
      </div>
      {
        snmp_receivers_loading ? (
          <div className="flex justify-center items-center h-48">
            <Loader2Icon className="animate-spin" />
          </div>
        ) : (
          snmp_receivers && <DataTable columns={alertingSnmpReceiversColumns} data={snmp_receivers || []} />
        )
      }
    </div>
  );
};

const AlertingCommands = ({ t }: { t: TFunction }) => {
  const { data: commands, isLoading: commandsLoading } = useGetCommands();
  const { mutate: deleteCommand } = useDeleteCommand();
  const { user } = useAuth();

  const alertingCommandsColumns = [
    {
      accessorKey: 'name',
      header: 'Name',
    },
    {
      accessorKey: 'location',
      header: 'Script Location',
    },
    {
      accessorKey: 'script',
      header: 'Script Name',
    },
    {
      accessorKey: 'arguments',
      header: 'Passed Args',
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => (
        <div className="hidden" />
      ),
      cell: ({ row }: { row: Row<any> }) => {
        const [dropdownOpen, setDrodownOpen] = useState(false);

        return (
          <div className="flex justify-end">
            {!!user?.is_admin &&
              <DropdownMenu open={dropdownOpen} onOpenChange={setDrodownOpen}>
                <DropdownMenuTrigger asChild>
                  <Button size="sm" variant="ghost" className="gap-2 text-xs">
                    <MoreHorizontalIcon />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  {
                    commands && (
                      <EditCommandModal command={commands.filter((command: Command) => command.id == row.original.id)[0]} onModalClose={() => setDrodownOpen(false)} />
                    )
                  }
                  <DropdownMenuItem onClick={() => deleteCommand(row.original.id)}>
                    <Trash2Icon />
                    <span className="hidden lg:block">{t(`Delete`)}</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            }
          </div>
        )
      },
    }
  ];

  return (
    <div className="flex flex-col gap-4">
      <div className="flex gap-2 w-full justify-between">
        <div className="flex flex-col gap-1">
          <h1 className="text-xl font-medium flex items-center gap-1.5">
            <SquareTerminalIcon size={20} />
            {t(`Commands`)}
          </h1>
          <h3 className="text-sm text-muted-foreground">
            {t(`Define custom scripts available for checks to use.`)}
          </h3>
        </div>

        <AddCommandModal />
      </div>
      {
        commandsLoading ? (
          <div className="flex justify-center items-center h-48">
            <Loader2Icon className="animate-spin" />
          </div>
        ) : (
          commands && <DataTable columns={alertingCommandsColumns} data={commands || []} defaultColumnVisibility={{
            arguments: false,
          }} />
        )
      }
    </div>
  );
};
