import {
  PageLayout,
  PageHeader,
  PageTitle,
  PageSeparator,
} from "@/components/Page";
import {
  Card,
  CardContent,
} from "@/components/ui/card"
import { useState } from "react";
import { Link } from "react-router-dom";
import { useTranslation } from "react-i18next";
import { useGetSources } from "@/components/source/queries";
import { useGetSourceGroupSummaryChart } from "@/components/sourcegroups/queries";
import { useGetNmapVersion } from "@/components/nmap/queries";
import { useGetNmapOverview } from "@/components/nmap/overview/queries";
import { NmapScansOverTimeBarChart } from "@/components/nmap/overview/ScansOverTimeBarChart";
import { SummaryChart } from "@/components/source/SummaryChart";
import { useCheckSuricata } from "@/components/suricata/queries";
import { useGetWiresharkVersion } from "@/components/wireshark/queries";
import { WiresharkScansOverTimeBarChart } from "@/components/wireshark/overview/ScansOverTimeChart";
import { IntegrationNotInstalled } from "@/components/home/IntegrationNotInstalled";
import { CategoryChart } from "@/components/suricata/overview/CategoryChart";
import { Button } from "@/components/ui/button";

export default function Home(){
  const {data:sourceData, isLoading: sourceIsLoading } = useGetSources();
  const { data: summaryChartflowGroup, isLoading: isSummaryLoading } = useGetSourceGroupSummaryChart(1, "-24 hour");
  const { data: nmapVersion, isLoading: nmapVersionLoading } = useGetNmapVersion();
  const { data: nmapOverviewData } = useGetNmapOverview();
  const { data: suricataVersion, isLoading: suricataVersionLoading, } = useCheckSuricata();
  const { data: wiresharkVersion,  isLoading: wiresharkVersionLoading} = useGetWiresharkVersion();
  const {t} = useTranslation();

  const [isLogarithmic, setIsLogarithmic] = useState(true);
  
  // Function to get the appropriate unit
  const getBytesUnit = (bytes: number) => {
    if (bytes === 0) return 'B';
    
    const k = 1000;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return sizes[i];
  };
  
  const rawChartData = summaryChartflowGroup?.chart_data || [];
  const bytesUnit = rawChartData.length > 0 ? getBytesUnit(rawChartData[0].bytes) : 'B';
  return(
    <PageLayout>
      <PageHeader>
        <PageTitle>
          {t("Home")}
        </PageTitle>
      </PageHeader>
      <div className="grid gap-4 mt-4" style={{ gridTemplateColumns: "repeat(auto-fit, minmax(603px, 1fr))" }}>
      <Card className="h-[450px]">
        <CardContent className="h-full flex flex-col">
          {sourceIsLoading ? (
            <p>{t("Loading sources...")}</p>
          ) : !sourceData || sourceData.length === 0 ? (
            <>
              <h3 className="text-lg">
                {t("Flow Sources")} 
              </h3>
              <PageSeparator/>
              <div className="flex flex-col justify-between h-full">
                <p className="text-lg">{t("Flow data helps you understand network usage, detect anomalies, and monitor long-term trends.")}</p>
                <div>
                  <p className="font-medium mb-2 text-lg">{t("Key benefits")}:</p>
                  <ul className="list-disc list-inside space-y-2 text-md">
                    <li>{t("Identify top talkers and applications consuming bandwidth.")}</li>
                    <li>{t("Detect anomalies and unusual traffic patterns for security.")}</li>
                    <li>{t("Track historical usage trends for capacity planning.")}</li>
                  </ul>
                </div>
                <Link className="text-primary" to={"/sources"}>
                <Button>{t("Set up Flow Sources now")}</Button>
                </Link>
              </div>
            </>
          ) : (
            <>
            <div className="flex w-full justify-between items-center">
              <h3 className="text-lg font-semibold">{t("All Flow Sources - Last 24 Hours")}</h3>
              <div className="flex rounded-md border border-muted overflow-hidden p-1 gap-1">
                <button
                  onClick={() => setIsLogarithmic(false)}
                  className={`px-3 py-1 text-sm font-medium transition-colors w-[120px] rounded-md transition-all duration-300 ease-out ${
                    !isLogarithmic 
                      ? 'bg-primary text-primary-foreground' 
                      : 'bg-card text-muted-foreground hover:bg-muted'
                  }`}
                >
                  {t('Linear')}
                </button>
                <button
                  onClick={() => setIsLogarithmic(true)}
                  className={`px-3 py-1 text-sm font-medium transition-colors w-[120px] rounded-md transition-all duration-300 ease-out ${
                    isLogarithmic 
                      ? 'bg-primary text-primary-foreground' 
                      : 'bg-card text-muted-foreground hover:bg-muted'
                  }`}
                >
                  {t('Logarithmic')}
                </button>
              </div>
            </div>
            <PageSeparator />
            <SummaryChart 
              data={rawChartData} 
              isLoading={isSummaryLoading} 
              isLogarithmic={isLogarithmic}
              bytesUnit={bytesUnit}
            />
            </>
          )}
        </CardContent>
      </Card>
      <Card className="h-[450px]">
        <CardContent className="h-full flex flex-col">
          {nmapVersionLoading ? (
            <p>{t("Checking for Nmap...")}</p>
          ) : !nmapVersion || nmapVersion.nmap_version === "None" ? (
            <IntegrationNotInstalled
              name="Nmap"
              description={t("Nmap provides powerful network scans to reveal hosts, services, and potential risks across your network.")}
              bullets={[
                t("Discover hosts and services across your network with fast, accurate scans."),
                t("Schedule recurring scans to maintain up-to-date visibility."),
                t("Detect new or removed devices with ndiff comparisons."),
              ]}
              link="/nmap"
              t={t}
            />
          ) : (
            <>
              <h3 className="text-lg font-semibold">Nmap</h3>
              <PageSeparator />
              <NmapScansOverTimeBarChart data={nmapOverviewData?.scan_in_last7days ?? []} />
            </>
          )}
        </CardContent>
      </Card>
      <Card className="h-[450px]">
        <CardContent className="h-full flex flex-col">
        {suricataVersionLoading ? (
            <p>{t("Checking for Suricata...")}</p>
          ) : !suricataVersion || suricataVersion.version === "None" ? (
            <IntegrationNotInstalled
            name="Suricata"
            description={t("Suricata delivers real-time intrusion detection to spot malware, exploits, and policy violations.")}
            bullets={[
              t("Detect malware, exploits, and policy violations in real time."),
              t("Send alerts to designated recipients for faster response."),
              t("Support investigations with detailed logs and metadata."),
            ]}
            link="/suricata"
            t={t}
          />

          ) : (
            <>
              <h3 className="text-lg font-semibold">Suricata</h3>
              <PageSeparator />
              <div className="h-[338px]">
                <CategoryChart />
              </div>
            </>
          )}
        </CardContent>
      </Card>
      <Card className="h-[450px]">
        <CardContent className="h-full flex flex-col">
        {wiresharkVersionLoading ? (
            <p>{t("Checking for Wireshark...")}</p>
          ) : !wiresharkVersion || wiresharkVersion === "None" ? (
            <IntegrationNotInstalled
              name="Wireshark"
              description="Wireshark captures and analyzes packets for troubleshooting and forensic investigation."
              bullets={[
                "Capture and inspect packets for in-depth troubleshooting.",
                "Trigger targeted packet captures directly from Suricata alerts for deeper analysis.",
                "Support investigations with saved packet captures."
              ]}
              link="/wireshark"
              t={t}
            />
          ) : (
            <>
              <h3 className="text-lg font-semibold">Wireshark</h3>
              <PageSeparator />
              <WiresharkScansOverTimeBarChart />
            </>
          )}
        </CardContent>
      </Card>
      </div>
    </PageLayout>
  )
}