import { useTranslation } from 'react-i18next';
import { RadioGroup } from '@radix-ui/react-radio-group';
import { RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { Button } from "@/components/ui/button"
import { ChevronRight } from 'lucide-react';
import { Input } from '@/components/ui/input';
import { TimeZoneDropdown } from '@/components/administration/global/TimeZoneDropdown';
import LanuageDropdown from '@/components/ui/LanguageDropdown';
import ThemesDropdown from '@/components/ui/ThemesDropdown';
import { useForm } from 'react-hook-form'
import { PasswordInput } from '@/components/ui/passwordInput';
import { useMutation } from '@tanstack/react-query';
import { useNavigate } from 'react-router-dom';
import {
    Form,
    FormControl,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"
import {
    PageDescription,
    PageSeparator,
  } from "@/components/Page";
import axios from 'axios';
import { toast } from 'sonner';
import { useQueryClient } from "@tanstack/react-query";

type StringObject = {
[key: string]: string;
};

// const FormSchema = z.object({
// email: z
//     .string({
//     required_error: "Please select an email to display.",
//     })
//     .email(),
// userName: z
//     .string({
//     required_error: "Please input a user name."
//     })
// })

export default function Install() {
    const { t } = useTranslation();
    const navigate = useNavigate();
    const form = useForm({
        defaultValues: {
            licenseOption: "trial",
            licenseKey: "",
            userName: "nagiosadmin",
            password: "",
            passwordConfirm: "",
            email: "",
            language: "en_US",
            timezone: "",
            theme: "default",
        }
    });
    const licenseOption = form.watch("licenseOption");
    const queryClient = useQueryClient();

    const addUser = (data: StringObject) => {
        return axios.post('/api/v1/install', data);
    }
    
    const addInitialUser = useMutation({
        mutationFn: addUser,
        onSuccess: (response) => {
            queryClient.invalidateQueries({ queryKey: ['installed'] });
            queryClient.invalidateQueries({ queryKey: ['trial'] });
            queryClient.invalidateQueries({ queryKey: ['license'] });
            toast(t(response.data.message));
            navigate(response.data.redirect);
        },
        onError: () => {
            toast(t('Failed to create user.'));
        }
    })

    function onSubmit(values: StringObject) {
        addInitialUser.mutate(values)
    }

    return (
    <div className="mx-auto max-w-4xl w-full p-4">
        <h2 className="font-medium text-2xl mb-3">{t('Final Installation Steps')}</h2>
        <PageDescription>{t('Almost done! Update the license and configure the admin account.')}</PageDescription>
        <PageSeparator/>
        <Form {...form}>
            <form onSubmit={form.handleSubmit((onSubmit))}>
                <div className="pb-8">
                    <h3 className="font-medium text-2xl mb-3">{t('License Setup')}</h3>
                    <PageDescription>{t('Choose a trial license, enter your key, or ')}
                        <a className="text-primary" href="https://www.nagios.com/products/nagios-network-analyzer/" target="_blank">{t('get a license now')}</a>
                        .
                    </PageDescription>
                    <PageSeparator/>
                    <div className="flex flex-col gap-4">
                        <FormField 
                            control={form.control}
                            name="licenseOption"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormControl>
                                        <RadioGroup
                                            defaultValue="trial"
                                            className="gap-[.5rem] grid"
                                            onValueChange={field.onChange}  
                                            value={field.value}
                                        >
                                            <div className="flex items-center space-x-2">
                                                <RadioGroupItem value="trial" id="trial" />
                                                <Label htmlFor="trial">{t('Free 30 day trial')}</Label>
                                            </div>
                                            <div className="flex items-center space-x-2">
                                                <RadioGroupItem value="key" id="key" />
                                                <Label htmlFor="key">{t('I already have a key')}</Label>
                                            </div>
                                        </RadioGroup>
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField 
                            control={form.control}
                            name="licenseKey"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('License Key')}</FormLabel>
                                    <FormControl className="col-span-2">
                                        <Input 
                                            {...field}
                                            className={licenseOption === "option-one" ? "cursor-not-allowed" : ""}
                                            disabled={licenseOption === "option-one"}
                                        />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                    </div>
                </div>

                <div className="pb-8">
                    <h3 className="font-medium text-2xl mb-3">{t('Admin Account Setup')}</h3>
                    <PageDescription>{t('Enter the admin account settings. The default username is nagiosadmin.')}</PageDescription>
                    <PageSeparator/>
                    <div className="flex flex-col gap-4">
                        <FormField 
                            control={form.control}
                            name="userName"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('Username')}</FormLabel>
                                    <FormControl className="col-span-2">
                                        <Input {...field} />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField 
                            control={form.control}
                            name="password"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('Password')}</FormLabel>
                                    <div className="col-span-2">
                                        <FormControl className="col-span-2">
                                            <PasswordInput {...field} />
                                        </FormControl>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField 
                            control={form.control}
                            name="passwordConfirm"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('Confirm Password')}</FormLabel>
                                    <div className="col-span-2">
                                        <FormControl className="col-span-2">
                                            <PasswordInput {...field} />
                                        </FormControl>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="email"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('Email Address')}</FormLabel>
                                    <FormControl className="col-span-2">
                                        <Input {...field} placeholder="example@email.com" />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />    
                        <FormField
                            control={form.control}
                            name="language"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('Language')}</FormLabel>
                                    <FormControl className="col-span-2">
                                        <LanuageDropdown field={field} />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        {/* TODO: Add logic to select the timezone automatically if the timzone is already set. */}
                        <FormField
                            control={form.control}
                            name="timezone"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('System Timezone')}</FormLabel>
                                    <FormControl className="col-span-2">
                                        <div className="col-span-2 w-full">
                                        <TimeZoneDropdown {...field} />
                                        </div>
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="theme"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                                    <FormLabel className="text-sm font-medium">{t('User Theme')}</FormLabel>
                                    <FormControl className="col-span-2">
                                        <ThemesDropdown field={field} />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                    </div>
                    <div className="flex justify-end mt-8">
                        <Button className="bg-primary" type="submit">
                            {t('Finish Installation')}
                            <ChevronRight />
                        </Button>
                    </div>
                </div>
            </form>
        </Form>
    </div>
    );
  }