import {
  PageLayout,
  PageHeader,
  PageTitle,
  PageDescription,
  PageSeparator,
} from "@/components/Page";
import { 
  useGetSourceGroups,
  useStartSourceGroup,
  useStopSourceGroup,
  useRestartSourceGroup,
} from "@/components/sourcegroups/queries";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader"
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { SourceGroup } from "@/components/sourcegroups/types";
import { ColumnDef, Row } from "@tanstack/react-table";
import { Button } from "@/components/ui/button";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Ellipsis, Plus, Pencil, Trash2, CirclePower, RotateCcw, CirclePause } from "lucide-react";
import { useTranslation } from "react-i18next";
import { Link } from "react-router-dom";
import { useState } from "react";
import AddEditSourceGroupDialog from "@/components/sourcegroups/AddEditSourceGroupDialog";
import DeleteSourceGroupsDialog from "@/components/sourcegroups/DeleteSourceGroupsDialog";
import { Checkbox } from "@/components/ui/checkbox"
import { Badge } from "@/components/ui/badge";
import { useAuth } from "@/contexts/AuthContext";

export default function SourceGroups() {
  const { t } = useTranslation();
  const { data, isLoading } = useGetSourceGroups();
  const { user } = useAuth();

  // State for managing the edit dialog
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [editingSourceGroup, setEditingSourceGroup] = useState<SourceGroup | undefined>(undefined);

  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [idsToDelete, setIdsToDelete] = useState<number[]>([]);

  const startSourceGroupMutation = useStartSourceGroup();
  const stopSourceGroupMutation = useStopSourceGroup();
  const restartSourceGroupMutation = useRestartSourceGroup();

  const handleDelete = (id: number) => {
    setIdsToDelete([id]);
    setDeleteDialogOpen(true);
  }

  const parseDiskUsage = (val: string) => {
    if (!val || val === '0') return 0;
    const num = parseFloat(val);
    if (val.toUpperCase().includes('M')) return num * 1024 * 1024;
    if (val.toUpperCase().includes('G')) return num * 1024 * 1024 * 1024;
    return num;
  };
  
  const formatBytes = (bytes: number) => {
    if (bytes === 0) return '0';
    const units = ['B', 'K', 'M', 'G', 'T'];
    const i = Math.floor(Math.log(bytes) / Math.log(1024));
    return (bytes / Math.pow(1024, i)).toFixed(1) + units[i];
  };

  const actions = [
    {
      label: "Delete",
      value: "delete",
      onClick: async (selectedRows: any) => {
        setIdsToDelete(selectedRows.map((row: any) => row.id));
        setDeleteDialogOpen(true);
      }
    }
  ]

  const columns: ColumnDef<SourceGroup>[] = [
    {
      id: "select",
      header: ({ table }) => (
        <Checkbox
          checked={
            table.getIsAllPageRowsSelected() ||
            (table.getIsSomePageRowsSelected() && "indeterminate")
          }
          onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
          aria-label="Select all"
        />
      ),
      cell: ({ row }) => (
        <Checkbox
          checked={row.getIsSelected()}
          onCheckedChange={(value) => row.toggleSelected(!!value)}
          aria-label="Select row"
        />
      ),
      enableSorting: false,
      enableHiding: false,
    },
    {
      accessorKey: 'name',
      header: ({ column }) => {
        return <DataTableColumnHeader column={column} title="Name" />
      },
      cell: ({ row }) => {
        return <Link className="text-primary" to={'/groups/' + row.original.id}>{row.original.name}</Link>
      }
    },
    {
      id: 'Source Count',
      accessorKey: "source_count",
      sortingFn: (rowA, rowB) => {
        const sourcesA = rowA.original.sources || [];
        const sourcesB = rowB.original.sources || [];
        return sourcesA.length - sourcesB.length;
      },
      header: ({ column }) => (
        <DataTableColumnHeader column={column} title="Source Count" />
      ),
      cell: ({ row }) => {
        const sources = row.original.sources || [];
        return (
          <div>
            {sources.length}
          </div>
        );
      },
    },
    {
      accessorKey: 'diskusage',
      id: 'diskusage',
      header: ({ column }) => (
        <DataTableColumnHeader column={column} title="Disk" />
      ),
      cell: ({ row }) => {
        const totalBytes = (row.original.sources || []).reduce((acc: number, source) => {
          return acc + parseDiskUsage(source.diskusage ?? "");
        }, 0);

        return (
          <span>{totalBytes > 0 ? formatBytes(totalBytes) : 'No Data'}</span>
        );
      },
      sortingFn: (rowA: Row<SourceGroup>, rowB: Row<SourceGroup>): number => {
        const getTotalDiskBytes = (row: Row<SourceGroup>): number => {
          const groupSources = row.original.sources || [];
          return groupSources.reduce(
            (acc: number, source) => acc + parseDiskUsage(source.diskusage ?? ""),
            0
          );
        };

        const a = getTotalDiskBytes(rowA);
        const b = getTotalDiskBytes(rowB);
        return a - b;
      }
    },
    {
      accessorKey: 'sources',
      header: ({ column }) => (
        <DataTableColumnHeader column={column} title="Sources" />
      ),
      cell: ({ row }) => {
        const sources = row.original.sources || [];
        return (
          <div className="flex flex-wrap gap-2">
            {sources.map((source) => (
              <Link key={source.id} to={`/sources/${source.id}`}>
                <Badge
                  variant={source.status ? "green" : "red"}
                  className="cursor-pointer hover:underline"
                >
                  {source.name}
                </Badge>
              </Link>
            ))}
          </div>
        );
      },
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => (
        <div>
          {!!user?.is_admin &&
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost">
                  <Ellipsis />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent>
                {row.original.sources?.some(source => !source.status) && (
                  <DropdownMenuItem onClick={() => startSourceGroupMutation.mutate(row.original.id)}>
                    <CirclePower />
                    {t('Start All Sources')}
                  </DropdownMenuItem>
                )}
                {row.original.sources?.some(source => source.status) && (
                  <DropdownMenuItem onClick={() => stopSourceGroupMutation.mutate(row.original.id)}>
                    <CirclePause />
                    {t('Stop All Sources')}
                  </DropdownMenuItem>
                )}
                <DropdownMenuItem onClick={() => restartSourceGroupMutation.mutate(row.original.id)}>
                  <RotateCcw />
                  {t('Restart All Sources')}
                </DropdownMenuItem>
                {/* Hide Edit and Delete options for 'All Flow Sources' source group */}
                {row.original.id !== 1 && (
                  <>
                    <DropdownMenuItem 
                      onClick={() => {
                        setEditingSourceGroup(row.original);
                        setEditDialogOpen(true);
                      }}
                    >
                      <Pencil />
                      {t('Edit')}
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={() => handleDelete(row.original.id)}>
                      <Trash2 />
                      {t('Delete')}
                    </DropdownMenuItem>
                  </>
                )}
              </DropdownMenuContent>
            </DropdownMenu>
          }
        </div>
      ),
    },
  ];

  return (
    <PageLayout>
      <PageHeader>
        <PageTitle>{t("Flow Source Groups")}</PageTitle>
        <PageDescription>{t("Full list of flow source groups in your system. Flow source groups can consist of multiple sources and act like a single source for data accounting purposes and one source can be in multiple flow source groups")}.</PageDescription>
        <PageSeparator />
      </PageHeader>
      {!!user?.is_admin &&
        <AddEditSourceGroupDialog
          sourcegroup={editingSourceGroup}
          open={editDialogOpen}
          onOpenChange={(open) => {
            setEditDialogOpen(open);
            if (!open) {
              setEditingSourceGroup(undefined);
            }
          }}
        >
          <Button 
            className="mb-4"
            onClick={() => setEditingSourceGroup(undefined)}
          >
            <Plus />
            {t('Create Flow Source Group')}
          </Button>
        </AddEditSourceGroupDialog>
      }
      {isLoading && (
        <DataTableSkeleton></DataTableSkeleton>
      )}
      {!isLoading && (
        <DataTable 
          columns={columns} 
          data={data || []}
          usesRowSelection={true}
          selectedRowActions={actions}
        />
      )}
      <DeleteSourceGroupsDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      >
        <span />
      </DeleteSourceGroupsDialog>
    </PageLayout>
  );
}