#!/usr/bin/env python3

import subprocess
import logging
import os
import re
import config
import logging

try:
    from shlex import quote
except ImportError:
    import pipes
    quote = pipes.quote

logger = logging.getLogger('nnabackend')

def start(name, directory, port, flowtype, lifetime, sid):
    
    global logger
    
    #~ Initialize the directory structure, if it is already initialized
    #~ properly this is a noop
    initcmd = [ '/usr/local/nagiosna/scripts/initialize_source.py',
                '--directory', quote(directory),
                '--port', quote(str(port)) ]
    init = subprocess.Popen(' '.join(initcmd), shell=True, stdout=subprocess.PIPE)
    init.wait()
    
    if init.returncode != 0:
        logger.error('Error running initialization.')
    
    # jflow and ipfix don't have their own capd command and just use nfcapd
    if flowtype == 'netflow' or flowtype == 'jflow' or flowtype == 'ipfix':
        cmd = ['/usr/local/bin/nfcapd']
    elif flowtype == 'sflow':
        cmd = ['/usr/local/bin/sfcapd']
    else:
        raise Exception('Unknown type of flow collector %s for %s' % (type, name))
    
    pidfile = os.path.join(directory, '%d.pid' % port)
    
    if os.path.isfile(pidfile):
        try:
            pfile = open(pidfile)
            pid = int(pfile.readline())
            killing = subprocess.Popen(['sudo', '-u', 'nna', '/bin/kill', str(pid)], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            killing.wait()
            logger.warning(killing.stdout.readlines())
            os.remove(pidfile)
        except Exception as e:
            logger.info('Pidfile already exists at %s. If trouble continues, this pidfile will have to be manually removed.' % pidfile)
            print('Error reading from pidfile to kill existing process.')
            return
    
    sid = str(sid)
    
    cmd += ['-I', sid]
    #~ Specify our base directory we will be storing flows in.
    cmd += ['-w', os.path.join(directory, 'flows')]
    #~ Specify the port that we will be listening on.
    cmd += ['-p', str(port)]
    #~ Specify the script that will be called whenever the capd rotates its log
    cmd += ['-x', '/usr/local/nagiosna/scripts/reap_files.py %d %f %i']
    #~ Specify our pidfile
    cmd += ['-P', pidfile]
    #~ Make sure we daemonize and tell it to auto-expire flows and compress.
    cmd += ['-D', '-e', '-z=lzo']
    
    expire = ['/usr/local/bin/nfexpire', '-t', lifetime, '-u', os.path.join(directory, 'flows')]
    
    nfexpire = subprocess.Popen(expire, stdout=subprocess.PIPE)
    nfexpire.wait()
    nfcapd = subprocess.Popen(cmd)
    nfcapd.wait()
    
    if nfcapd.returncode != 0:
        logger.error('Could not start nfcapd listening on %d for source %s. Command: %s' % (port, name, ' '.join(cmd)))
    
def stop(name, directory, port, flowtype, lifetime, sid):
    
    global logger
    
    pidfile = '%s/%d.pid' % (directory, port)
    
    try:
        pidf = open(pidfile)
        try:
            pid = int(pidf.readline())
        except Exception as e:
            raise Exception('Could not stop %s, pidfile was malformed' % name)
                
        killing = subprocess.Popen(['sudo', '-u', 'nna', '/bin/kill', str(pid)], stdout=subprocess.PIPE)
        killing.wait()
        if not [] == killing.stdout.readlines():
            print('Error stopping pid %s' % str(pid))
            EXIT_CODE = 1
    except Exception as e:
        logger.info('Could not access %s for killing: %s' % (pidfile, str(e)))
